<?php

use App\Exceptions\ShopifyProductCreatorException;
use App\Lib\AuthRedirection;
use App\Lib\EnsureBilling;
use App\Lib\ProductCreator;
use App\Models\Session;
use App\Models\PersonalisedGiftAppHowToPageContent;
use App\Traits\PersonalisedAppTrait;
use App\Models\User;
use App\Models\PersonalisedGiftAppUser;
use App\Models\PersonalisedGiftAppFont;
use App\Models\PersonalisedGiftAppClipart;
use App\Models\PersonalisedGiftAppProduct;
use App\Models\PersonalisedGiftAppCategory;
use App\Models\PersonalisedGiftAppFaqPageContent;
use App\Models\ShopifyMetafield;
use App\Models\Job;
use App\Models\PersonalisedGiftAppProductMeta;
use App\Models\PersonalisedGiftAppUserProductRanges;
use App\Models\PersonalisedGiftAppCategoryProduct;
use App\Models\ShopifyStoreWebhook;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Route;
use Shopify\Auth\OAuth;
use Shopify\Auth\Session as AuthSession;
use Shopify\Clients\HttpHeaders;
use Shopify\Clients\Rest;
use Shopify\Context;
use Shopify\Exception\InvalidWebhookException;
use Illuminate\Support\Facades\Response;
use Shopify\Utils;
use Shopify\Webhooks\Registry;
use Shopify\Webhooks\Topics;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| contains the "web" middleware group. Now create something great!
|
| If you are adding routes outside of the /api path, remember to also add a
| proxy rule for them in web/frontend/vite.config.js
|
*/
Route::fallback(function (Request $request) {
    if (Context::$IS_EMBEDDED_APP &&  $request->query("embedded", false) === "1") {
        if (env('APP_ENV') === 'production') {
            return file_get_contents(public_path('index.html'));
        } else {
            return file_get_contents(base_path('frontend/index.html'));
        }
    } else {
        Log::channel('pgs-app-status')->info("embedded fallback function for the request info -".json_encode($request->all())." and the header info - ".json_encode($request->header())." and the query info - ".json_encode($request->query())." and path name - ".json_encode($request->path()));
        
        if(!empty($request->query('host'))){
            $hostUrl = $request->query("host", null);
            Log::channel('pgs-app-status')->info("Condition 1");
        }else{
            // Encode the host using base64
            $hostUrl = base64_encode($_ENV['HOST']);
            Log::channel('pgs-app-status')->info("Condition 2");
        }

        Log::channel('pgs-app-status')->info("host url - ".json_encode($hostUrl));
        // return redirect(Utils::getEmbeddedAppUrl($request->query("host", null)) . "/" . $request->path()); // old line 
        return redirect(Utils::getEmbeddedAppUrl($hostUrl) . "/" . $request->path());
    }
})->middleware('shopify.installed');


Route::get('/api/auth', function (Request $request) {
    $shop = Utils::sanitizeShopDomain($request->query('shop'));

    // Delete any previously created OAuth sessions that were not completed (don't have an access token)
    Session::where('shop', $shop)->where('access_token', null)->delete();

    return AuthRedirection::redirect($request);
});


Route::get('/api/auth/callback', function (Request $request) {
    $session = OAuth::callback(
        $request->cookie(),
        $request->query(),
        ['App\Lib\CookieHandler', 'saveShopifyCookie']
    );

    $host = $request->query('host');
    $shop = Utils::sanitizeShopDomain($request->query('shop'));

    $response = Registry::register('/api/webhooks', Topics::APP_UNINSTALLED, $shop, $session->getAccessToken());
    if ($response->isSuccess()) {
        Log::channel('pgs-app-status')->info("Registered APP_UNINSTALLED webhook for shop $shop");
    } else {
        Log::channel('pgs-app-status')->info(
            "Failed to register APP_UNINSTALLED webhook for shop $shop with response body: " .
                print_r($response->getBody(), true)
        );
    }

    // check theme if supports is 2.0
    $checkTheme = App\Http\Helpers\CustomHelpers::checkShopifyThemeSupport($shop, $session->getAccessToken()); 
    $veriyThemeStatus = 0;
    if(!empty($checkTheme) && count($checkTheme) > 0){
        $veriyThemeStatus = 1;
    }

    $sessionInfo = Session::where('shop', $shop)->where('access_token', $session->getAccessToken())->first();
    if(!empty($sessionInfo)){
        $sessionInfo->update(['theme_status' => $veriyThemeStatus]);
    }

    $redirectUrl = Utils::getEmbeddedAppUrl($host);
    if (Config::get('shopify.billing.required')) {
        list($hasPayment, $confirmationUrl) = EnsureBilling::check($session, Config::get('shopify.billing'));

        if (!$hasPayment) {
            $redirectUrl = $confirmationUrl;
        }
    }

    return redirect($redirectUrl);
});


Route::get('/api/products/count', function (Request $request) {
    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active

    $client = new Rest($session->getShop(), $session->getAccessToken());
    $result = $client->get('products/count');

    return response($result->getDecodedBody());
})->middleware('shopify.auth');


Route::get('/api/products/create', function (Request $request) {
    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active

    $success = $code = $error = null;
    try {
        ProductCreator::call($session, 5);
        $success = true;
        $code = 200;
        $error = null;
    } catch (\Exception $e) {
        $success = false;

        if ($e instanceof ShopifyProductCreatorException) {
            $code = $e->response->getStatusCode();
            $error = $e->response->getDecodedBody();
            if (array_key_exists("errors", $error)) {
                $error = $error["errors"];
            }
        } else {
            $code = 500;
            $error = $e->getMessage();
        }

        Log::error("Failed to create products: $error");
    } finally {
        return response()->json(["success" => $success, "error" => $error], $code);
    }
})->middleware('shopify.auth');


Route::post('/api/webhooks', function (Request $request) {
    try {

        $topic = $request->header(HttpHeaders::X_SHOPIFY_TOPIC, '');

        Log::channel('uninstall-pgs-app-webhook-status')->info("request is processing ");

        // delete pgs app related product, collection and webhooks
        $payload = json_decode($request->getContent(), true);

        $storeDomain = $payload['domain'];

        $checkSession = Session::where('shop', $storeDomain)->first();

        Log::channel('uninstall-pgs-app-webhook-status')->info("unistall webhook shop name - ".$storeDomain);

        if(!empty($checkSession)){

            $shopifyAuthSessionID = $checkSession->id;
            $shopAccessToken = $checkSession->access_token;

            Log::channel('uninstall-pgs-app-webhook-status')->info("unistall webhook shop acess token - ".$shopAccessToken);
        }

        Log::channel('uninstall-pgs-app-webhook-status')->info("Uninstall app webhook header request params - ".json_encode($request->header()));

        Log::channel('uninstall-pgs-app-webhook-status')->info("Uninstall app webhook contents params - ".json_encode($request->getContent()));

        $response = Registry::process($request->header(), $request->getContent());

        Log::channel('uninstall-pgs-app-webhook-status')->info("Uninstall app webhook check app response - ".json_encode($response));

        if (!$response->isSuccess()) {
            Log::error("Failed to process '$topic' webhook: {$response->getErrorMessage()}");
            return response()->json(['message' => "Failed to process '$topic' webhook"], 500);
        }
        
    } catch (InvalidWebhookException $e) {
        Log::error("Got invalid webhook request for topic '$topic': {$e->getMessage()}");
        return response()->json(['message' => "Got invalid webhook request for topic '$topic'"], 401);
    } catch (\Exception $e) {
        Log::error("Got an exception when handling '$topic' webhook: {$e->getMessage()}");
        return response()->json(['message' => "Got an exception when handling '$topic' webhook"], 500);
    }
});


Route::get('/api/get-user-settings', function (Request $request) {
    
    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
    $shopify_auth_token = $session->getAccessToken();

    $sessionInfo = Session::select('theme_status')->where('shop', $session->getShop())->where('access_token', $shopify_auth_token)->first();
    if(!empty($sessionInfo) && $sessionInfo->theme_status == 1){

        $userData = PersonalisedGiftAppUser::where('shopify_auth_token', $shopify_auth_token)->first();
        $SHOPIFY_VERSION = $_ENV['SHOPIFY_VERSION'];
        if(!empty($userData)){

            $totalPgsFonts = $userData->total_pgs_fonts;
            $totalPgsCliparts = $userData->total_pgs_cliparts;

            $totalSendPgsFonts = $userData->total_send_shopify_files_pgs_fonts;
            $totalSendPgsCliparts = $userData->total_send_shopify_files_pgs_cliparts;

            $personaliseddData = [
                'id' => $userData->id,
                'shopify_auth_token' => $userData->shopify_auth_token,
                'username' => $userData->username,
                'api_key' => $userData->api_key,
                'pgs_app_status' => ($userData->personalised_app_user_status == 1) ? "success" : "warning",
                'pgs_app_status_message' => ($userData->personalised_app_user_status == 1) ? "Verification Successful" : "Invalid Details",
                'total_pgs_fonts' => $totalPgsFonts,
                'total_pgs_cliparts' => $totalPgsCliparts,
                'send_shopify_pgs_fonts' => $totalSendPgsFonts,
                'send_shopify_pgs_cliparts' => $totalSendPgsCliparts,
            ];
            $responseData = [
                'status' => true,
                'data' => $personaliseddData,
                'theme_status' => true
            ];
            
        }else{
            $responseData = [
                'status' => true,
                'data' => [],
                'theme_status' => true
            ];
        }
    }else{
        $responseData = [
            'status' => false,
            'data' => [],
            'theme_status' => false
        ];
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::post('/api/update-user-settings', function (Request $request) {
    /** @var AuthSession */

    $customFontsArray = [
        'Times New Roman' => $_ENV['APP_URL'].'fonts/TimesNewRoman.woff',
        'Zapf Chancery' => $_ENV['APP_URL'].'fonts/ZapfChancery.woff',
        'Vivaldi' => $_ENV['APP_URL'].'fonts/Vivaldi.woff',
        'Palace Script' => $_ENV['APP_URL'].'fonts/PalaceScript.woff',
        'Old English' => $_ENV['APP_URL'].'fonts/OldEnglish.woff',
        'Lobster-Regular' => $_ENV['APP_URL'].'fonts/Lobster-Regular.woff',
        'Harrington' => $_ENV['APP_URL'].'fonts/Harrington.woff',
        'GreatVibes-Regular' => $_ENV['APP_URL'].'fonts/GreatVibes-Regular.woff',
        'French Script' => $_ENV['APP_URL'].'fonts/French-Script.woff',
        'Edwardian Script' => $_ENV['APP_URL'].'fonts/Edwardian-Script.woff',
        'Curlz' => $_ENV['APP_URL'].'fonts/Curlz.woff',
        'Chicle Regular' => $_ENV['APP_URL'].'fonts/Chicle-Regular.woff',
        'Evil Empire' => $_ENV['APP_URL'].'fonts/EvilEmpire.woff',
        'MTCORSVA' => $_ENV['APP_URL'].'fonts/MTCORSVA.woff',
        'Eras' => $_ENV['APP_URL'].'fonts/Eras.woff',
        'Coppler Plate' => $_ENV['APP_URL'].'fonts/CopplerPlate.woff',
        'Cooper' => $_ENV['APP_URL'].'fonts/Cooper.woff',
        'Bauhaus' => $_ENV['APP_URL'].'fonts/Bauhaus.woff',
        'Trumpit' => $_ENV['APP_URL'].'fonts/Trumpit.woff',
        'popwarner' => $_ENV['APP_URL'].'fonts/PopWarner.woff',
    ];
    
    if(empty($request->username) && empty($request->apikey)){
        $responseData = [
            'status' => false,
            'data' => [],
            'message' => 'Please enter Username and Api Key'
        ];
    }else{
        $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
        $shopify_auth_token = $session->getAccessToken();
        $checkSession = Session::where('access_token', $shopify_auth_token)->first();

        $pgs_third_party_api_url = $_ENV['PERSONALISED_GIFT_APP_API_URL'];
        $pgs_third_party_product_api_url = $_ENV['PERSONALISED_GIFT_APP_PRODUCT_API_URL'];
        $customer_response = App\Traits\PersonalisedAppTrait::callPersonalisedAppAPI($pgs_third_party_api_url."user/token", "GET", $request->apikey, $request->username, "");
     
        $decodedUser = json_decode($customer_response);
        $userverified = 'Invalid User';
        $showMessage = 'User registerd successfully but user not verified on the personalised gift supply returning response Invalid user.';
        if(!empty($decodedUser) && $decodedUser->statusCode == 200 && $decodedUser->success == 'true'){
            $userverified = $decodedUser->message;
            $showMessage = 'User registerd successfully and user verified on the personalised gift supply';
        }

        $shopifyAuthSessionId = (!empty($checkSession)) ? $checkSession->id : NULL;
    
        $personalisedUserData = [
            'shopify_auth_session_id' => $shopifyAuthSessionId,
            'shop_domain' => $session->getShop(),
            'username' => $request->username,
            'api_key' => $request->apikey,
            'shopify_auth_token' => $shopify_auth_token,
            'personalised_app_user_status' => ($userverified != 'Invalid User') ? 1 : 0
        ];
    
        $checkUserDetails = PersonalisedGiftAppUser::where('username', $request->username)->where('api_key', $request->apikey)->where('shopify_auth_token', '!=', $shopify_auth_token)->where('shopify_auth_session_id', '!=', $personalisedUserData['shopify_auth_session_id'])->get();

        if($checkUserDetails->count() >= 1){
            $responseData = [
                'status' => false,
                'data' => [],
                'message' => 'These details already exists with another store'
            ];

        }else{

            $userData = PersonalisedGiftAppUser::updateOrCreate(['shop_domain' => $session->getShop()], $personalisedUserData);
            if(!empty($userData)){

                $SHOPIFY_VERSION = $_ENV['SHOPIFY_VERSION'];
                $totalPgsFonts = $userData->total_pgs_fonts;
                $totalPgsCliparts = $userData->total_pgs_cliparts;

                $totalSendPgsFonts = $userData->total_send_shopify_files_pgs_fonts;
                $totalSendPgsCliparts = $userData->total_send_shopify_files_pgs_cliparts;

                $shopName = $checkSession->shop;
                $shopAccessToken = $checkSession->access_token;
                $shopId = $userData->id;

                // check user pricing option if selected then update selected value otherwise update suggested rrp value
                if(!empty($userData->pricing_options)){
                    $pricingOption = $userData->pricing_options;
                }else{
                    $pricingOption = 'suggested_rrp';
                }
                
                // check user next day shipping option is enabled or not
                if(!empty($userData->next_day_shipping_type_value)){
                    $shippingOption = $userData->next_day_shipping_type_value;
                }else{
                    $shippingOption = 'disable_next_day_shipping';
                }

                // check automatic process order status
                if(!empty($userData->automatically_process_order)){
                    $processOrderStatus = $userData->automatically_process_order;
                }else{
                    $processOrderStatus = 1;
                }

                $updateAdditionalInfo['pricing_options'] = $pricingOption;
                $updateAdditionalInfo['next_day_shipping_type_value'] = $shippingOption;
                $updateAdditionalInfo['automatically_process_order'] = $processOrderStatus;

                PersonalisedGiftAppUser::where('id', $userData->id)->update($updateAdditionalInfo);

                // store shopify shop ID
                if(empty($userData->shopify_store_id)){
                    $shopifyStoreInfo = ProductCreator::getShopifyShopInformation($session);
                    if(!empty($shopifyStoreInfo->data)){
                        $shopify_store_id = $shopifyStoreInfo->data->shop->id;
                        $userData->update(['shopify_store_id' => $shopify_store_id]);
                    }
                }

                // store sales channels ID
                if(empty($userData->store_sales_channel_id)){

                    $salesChannelId = NULL;
                    $salesChannels = ProductCreator::getAvailableSalesChannels($session);
                    if(!empty($salesChannels)){
                        if(!empty($salesChannels->data)){
                            $salesChannelId = $salesChannels->data->publications->edges[0]->node->id;
                        }
                    }
                    $userData->update(['store_sales_channel_id' => $salesChannelId]);
                }

                // get & store active theme id 
                if(empty($userData->shopify_store_active_theme_id)){

                    $SHOPIFY_THEME_API = "https://$shopName/admin/api/$SHOPIFY_VERSION/themes.json";
					
                    $theme_response = App\Traits\PersonalisedAppTrait::callCurlShopifyRestAPI($SHOPIFY_THEME_API, "GET", '', $shopAccessToken);
                    $decoded_theme_response = json_decode($theme_response);
                    if(isset($decoded_theme_response->themes) && !empty($decoded_theme_response->themes)){
                        foreach($decoded_theme_response->themes as $theme)
                        {
                            if($theme->role == "main")
                            {
                                $active_theme_id = $theme->admin_graphql_api_id;
                                $userData->update(['shopify_store_active_theme_id' => $active_theme_id]);
                            }
                        }
                    }
                }
                
                $pgsUserId = $userData->id;

                if($userData->personalised_app_user_status == 1){

                    // create order webhooks on the store
                    $shopifyOrderUrl = $_ENV['APP_URL'].'api/createorderbywebhook';
                    $shopifyInventoryUrl = $_ENV['APP_URL'].'api/updateproductinventory';
                    $shopifyDeletedProductUrl = $_ENV['APP_URL'].'api/deleteproductbywebhook';
                    $shopifyUpdateProductUrl = $_ENV['APP_URL'].'api/updateproductbywebhook';

                    // create webhook using shopify REST API
                    $SHOPIFY_WEBHOOK_API = "https://$shopName/admin/api/$SHOPIFY_VERSION/webhooks.json";
                    
                    // create order webhooks on the store
                    $checkStoreOrderWebhooks = ShopifyStoreWebhook::where('personalised_gift_app_user_id', $pgsUserId)->where('topic', 'ORDERS_CREATE')->whereNotNull('webhook_id')->first();
                    if(empty($checkStoreOrderWebhooks)){
                        
                        App\Http\Helpers\CustomHelpers::createRestShopiyStoreHooks($SHOPIFY_WEBHOOK_API, $shopifyOrderUrl, 'orders/create', $pgsUserId, $shopAccessToken); // added code for create webhook on the shopify store using rest API
                    }

                    // create inventory webhooks on the store
                    $checkStoreInventoryWebhooks = ShopifyStoreWebhook::where('personalised_gift_app_user_id', $pgsUserId)->where('topic', 'INVENTORY_LEVELS_UPDATE')->whereNotNull('webhook_id')->first();
                    if(empty($checkStoreInventoryWebhooks)){

                        App\Http\Helpers\CustomHelpers::createRestShopiyStoreHooks($SHOPIFY_WEBHOOK_API, $shopifyInventoryUrl, 'inventory_levels/update', $pgsUserId, $shopAccessToken); // added code for create webhook on the shopify store using rest API
                    }
                    
                    // create update product webhooks on the store
                    $checkStoreDeletedProductWebhooks = ShopifyStoreWebhook::where('personalised_gift_app_user_id', $pgsUserId)->where('topic', 'PRODUCTS_UPDATE')->whereNotNull('webhook_id')->first();
                    if(empty($checkStoreDeletedProductWebhooks)){

                        App\Http\Helpers\CustomHelpers::createRestShopiyStoreHooks($SHOPIFY_WEBHOOK_API, $shopifyUpdateProductUrl, 'products/update', $pgsUserId, $shopAccessToken); // added code for create webhook on the shopify store using rest API
                    }
                    
                    // create delete product webhooks on the store
                    $checkStoreDeletedProductWebhooks = ShopifyStoreWebhook::where('personalised_gift_app_user_id', $pgsUserId)->where('topic', 'PRODUCTS_DELETE')->whereNotNull('webhook_id')->first();
                    if(empty($checkStoreDeletedProductWebhooks)){

                        App\Http\Helpers\CustomHelpers::createRestShopiyStoreHooks($SHOPIFY_WEBHOOK_API, $shopifyDeletedProductUrl, 'products/delete', $pgsUserId, $shopAccessToken); // added code for create webhook on the shopify store using rest API
                    }

                    // create fulfillment service for the store
                    if(empty($userData->store_location_id)){

                        $locationResponse = ProductCreator::getStoreLocation($session);
                        $is_location_added = '';
                        if(!empty($locationResponse)){
                            if(!empty($locationResponse->data->locations->edges)){

                                $shopifyLocationId = $locationResponse->data->locations->edges[0]->node->id;
                                $userData->update(['store_location_id' => $shopifyLocationId]);
                                $is_location_added = 'yes';
                            }
                        } 

                        if(empty($is_location_added)){
                            /* App\Http\Helpers\CustomHelpers::createShopifyStoreLocation($shopName, $shopAccessToken, $shopId); */   // create store location
                            App\Http\Helpers\CustomHelpers::createShopifyStoreFulfillmentService($shopName, $shopAccessToken, $shopId);    // create fulfillment service store location
                        }
                    }

                    if($request->type == 'sync_data'){
                        
                        $showMessage = "PGS app font and cliparts and list products data synced successfully"; 
                        // store PGS clipart data
                        App\Http\Helpers\CustomHelpers::storePgsClipartData($request->apikey, $request->username, $pgsUserId); 
                        
                        // store static global fonts
                        if(!empty($customFontsArray)){
                            foreach($customFontsArray as $fontkey => $customFont){
                                
                                $saveCustomFontData = [
                                    'personalised_gift_app_user_id' => $pgsUserId,
                                    'name_of_font' => $fontkey,
                                    'font_file' => $customFont,
                                    'font_type' => 'global_fonts',
                                ];
                                PersonalisedGiftAppFont::updateOrCreate(['personalised_gift_app_user_id' => $pgsUserId, 'name_of_font' => $fontkey, 'font_file' => $customFont, 'font_type' => 'global_fonts'], $saveCustomFontData);
                            }
                        }

                        // store PGS fonts data
                        App\Http\Helpers\CustomHelpers::storePgsFontData($request->apikey, $request->username, $pgsUserId); 

                        // get total pgs fonts and cliparts
                        $personalisedFonts = PersonalisedGiftAppFont::count();
                        $personalisedCliparts = PersonalisedGiftAppClipart::count();

                        $updateUserPgsInfodata = [
                            'total_pgs_fonts' => (!empty($personalisedFonts)) ? $personalisedFonts : 0,
                            'total_pgs_cliparts' => (!empty($personalisedCliparts)) ? $personalisedCliparts : 0,
                        ];

                        $userData->update($updateUserPgsInfodata); // update user font and cliparts data

                        // storee total PGS products
                        /* $total_pgs_product_response = App\Traits\PersonalisedAppTrait::callPersonalisedAppAPI($pgs_third_party_api_url."product/count", "GET", $request->apikey, $request->username, "");
                        
                        $totalPgsProducts = json_decode($total_pgs_product_response);
                        if(!empty($totalPgsProducts->data)){
                            $totalProduct = (int)$totalPgsProducts->data->count;
                            $userData->update(['total_pgs_products' => $totalProduct]);
                        } 

                        if(!empty($userData->price_range_values)){
                            // App\Http\Helpers\CustomHelpers::getPgsTotalPaginationRecords($shopifyAuthSessionId, $request->apikey, $request->username);
                        } */

                        App\Http\Helpers\CustomHelpers::createShopStoreMetafields($userData->shopify_auth_token);  
    
                        /* 
                        // store products data
                        $pgs_product_response = App\Traits\PersonalisedAppTrait::callPersonalisedAppAPI($pgs_third_party_api_url."product/list/", "GET", $request->apikey, $request->username, "");
            
                        $decodedPgsProducts = json_decode($pgs_product_response);
    
                        $shop_domain_name = NULL;
                        $shop_db_id = NULL;
                        if(!empty($checkSession)){
                            $shop_domain_name = $checkSession->shop;
                            $shop_db_id = $checkSession->id;
                        }
    
                        if(!empty($decodedPgsProducts) && $decodedPgsProducts->statusCode == 200 && $decodedPgsProducts->success == 'true' && !empty($decodedPgsProducts->data)){
                            
                            $productData = $decodedPgsProducts->data; // this is for list api

                            $userStoreInfo = PersonalisedGiftAppUser::where('id', $userData->id)->first();
                            $rangeType = $userStoreInfo->pricing_options;
                            $marginPrice = $userStoreInfo->margin_change;
                            $deductPrice = $userStoreInfo->deduct_price_change;

                            // store products data
                            $store_pgs_product_data = App\Http\Helpers\CustomHelpers::createPgsProductData($productData, $shop_domain_name, $shop_db_id, $rangeType, $marginPrice, $deductPrice, $userStoreInfo->price_range_values);   
                        }     
                        
                        App\Http\Helpers\CustomHelpers::updateTotalSyncProductCount($shopifyAuthSessionId);   */ // comment first product list data               
                    }
                }

                $personaliseddData = [
                    'id' => $userData->id,
                    'shopify_auth_token' => $userData->shopify_auth_token,
                    'username' => $userData->username,
                    'api_key' => $userData->api_key,
                    'pgs_app_status' => ($userData->personalised_app_user_status == 1) ? "success" : "warning",
                    'pgs_app_status_message' => ($userData->personalised_app_user_status == 1) ? "Verification Successful" : "Invalid Details",
                    'total_pgs_fonts' => $totalPgsFonts,
                    'total_pgs_cliparts' => $totalPgsCliparts,
                    'send_shopify_pgs_fonts' => $totalSendPgsFonts,
                    'send_shopify_pgs_cliparts' => $totalSendPgsCliparts,
                ];
    
                $responseData = [
                    'status' => true,
                    'data' => $personaliseddData,
                    'message' => $showMessage
                ];
                
            }else{
                $responseData = [
                    'status' => false,
                    'data' => [],
                    'message' => 'There was an error while registering users'
                ];
            }
        }
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::get('/api/get-order-setting', function (Request $request) {
    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
    $shopify_auth_token = $session->getAccessToken();

    $sessionInfo = Session::select('theme_status')->where('shop', $session->getShop())->where('access_token', $shopify_auth_token)->first();

    if(!empty($sessionInfo) && $sessionInfo->theme_status == 1){
        $userData = PersonalisedGiftAppUser::where('shopify_auth_token', $shopify_auth_token)->first();
        if(!empty($userData)){

            $orderSettingData = App\Http\Helpers\CustomHelpers::getOrderSettingData($userData->shopify_auth_session_id);  
                
            if(!empty($orderSettingData)){
                $responseData = [
                    'status' => true,
                    'data' => $orderSettingData,
                    'theme_status' => true
                ];
            }else{
                $responseData = [
                    'status' => false,
                    'data' => [],
                    'theme_status' => true
                ];
            }

        }else{
            $responseData = [
                'status' => true,
                'data' => [],
                'theme_status' => true
            ];
        }
    }else{
        $responseData = [
            'status' => false,
            'data' => [],
            'theme_status' => false
        ];
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::post('/api/update-order-settings', function (Request $request) {
    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
    $shopify_auth_token = $session->getAccessToken();
    
    $SHOPIFY_VERSION = $_ENV['SHOPIFY_VERSION'];
    $userData = PersonalisedGiftAppUser::where('shopify_auth_token', $shopify_auth_token)->first();

    if(!empty($userData)){
        
        $automatedJobStatus = App\Http\Helpers\CustomHelpers::getShopRunningJobStatus($userData->shop_domain);
        if($automatedJobStatus == 0){
            
            if(!empty($request->price_range_values)){
                
                $marginErrorMessage = '';
                if($request->pricing_options == 'custom_margin' && !empty($request->margin_change)){
                    if(!($request->margin_change > 0 && $request->margin_change < 99.9)){
                        $marginErrorMessage = 'Markup should be less than 99.9 and greater than 0 or valid Integer';
                    }
                }

                if(empty($marginErrorMessage)){
                    
                    // check product deduct pricing status
                    $added_margin_change = $userData->margin_change;
                    $added_deduct_price_change = $userData->deduct_price_change;
                    $added_price_range_values = $userData->price_range_values;
                    $added_pricing_options = $userData->pricing_options;
                    $userId = $userData->id;
                    
                    if($request->price_range_values == $added_price_range_values){
                        $lastSyncPageValue = $userData->last_sync_page;
                    }else{
                        $lastSyncPageValue = 0;
                    }
                    
                    $shopifyLocationId = '';
            		$shopifySalesChannelId = '';
            		if(!empty($userData->store_location_id)){
            			$shopifyLocationId = $userData->store_location_id;
            		}
            
            		if(!empty($userData->store_sales_channel_id)){
            			$shopifySalesChannelId = $userData->store_sales_channel_id;
            		}
                    
                    $productRangeTypesArray = ['personalised_products','decorated_glassware','engraved_trophies','say_it_with_glass'];
                    $optionTypes = array('set_own_price', 'suggested_rrp');
                    
                    $shopName = $userData->session->shop;
            		$shopAccessToken = $userData->session->access_token;
            		$shopifyAuthSessionId = $userData->shopify_auth_session_id;
            		
            		$rangeOption = $request->pricing_options;
                    $marginPrice = (!empty($request->margin_change)) ? $request->margin_change : 0.00;
                    $deductPrice = (!empty($request->deduct_price_change)) ? $request->deduct_price_change : 0.00;
                    // $importCategoryStatus = $request->import_category;
                    
                    $selectedRangeArray = [];
                    $unSelectedRangeArray = [];
                    $decodeRangeValue = explode(",", $request->price_range_values);
                    foreach($productRangeTypesArray as $rangeArray){
                        if(in_array($rangeArray, $decodeRangeValue)){
                            $selectedRangeArray[] = $rangeArray;
                        }else{
                            $unSelectedRangeArray[] = $rangeArray;
                        }
                    }
                    
                    $pgs_btn_status = 0;
                    $pgs_btn_selector = NULL;
                    if($request->set_personalised_button_position == 'true'){
                        $pgs_btn_status = ($request->set_personalised_button_position == 'true') ? 1 : 0;
                        $pgs_btn_selector = (isset($request->set_personalised_button_selector) && !empty($request->set_personalised_button_selector)) ? $request->set_personalised_button_selector : NULL;
                    }
                    
                    if(in_array($rangeOption, $optionTypes)){
                        $marginPrice = 0;
                        $deductPrice = 0;
                    }
            
                    $updateUserData = [
                        'sync_type_value' => $request->sync_type_value,
                        'next_day_shipping_type_value' => $request->next_day_shipping_type_value,
                        'automatically_process_order' => ($request->automatically_process_order == 'true') ? 1 : 0,
                        'process_shared_order' => ($request->process_shared_order == 'true') ? 1 : 0,
                        'import_category' => ($request->import_category == 'true') ? 1 : 0,
                        'import_all' => ($request->import_all == 'true') ? 1 : 0,
                        'set_personalised_button_position' => $pgs_btn_status,
                        'set_personalised_button_selector' => $pgs_btn_selector,
                        'pricing_options' => $rangeOption,
                        'margin_change' => $marginPrice,
                        'deduct_price_change' => $deductPrice,
                        'price_range_values' => $request->price_range_values,
                        'last_sync_page' => $lastSyncPageValue
                    ];
                    
                    $userData->update($updateUserData);
        
                    if(($added_margin_change != $marginPrice) || ($added_deduct_price_change != $deductPrice) || ($added_pricing_options != $rangeOption)){
                        if($rangeOption != 'set_own_price'){
                            App\Http\Helpers\CustomHelpers::updateProductPriceCrons($shopName, $rangeOption, $marginPrice, $deductPrice, $shopAccessToken, $shopifyAuthSessionId, $userId);
                        }
                    }
                    
                    if($userData->import_all == '1'){
                        App\Http\Helpers\CustomHelpers::publishProductsOnShopify($shopName, $rangeOption, $marginPrice, $deductPrice, $userId, $shopAccessToken, $shopifySalesChannelId, $shopifyLocationId, $shopifyAuthSessionId, $userData->import_category);
                    }
                    
                    if(!empty($selectedRangeArray) && count($selectedRangeArray) > 0){
                        App\Http\Helpers\CustomHelpers::UpdateShopifyProductStatus($shopName, $shopAccessToken, $userId, 'ARCHIVED', 'ACTIVE', $selectedRangeArray);
                    }
                    
                    if(!empty($unSelectedRangeArray) && count($unSelectedRangeArray) > 0){
                        App\Http\Helpers\CustomHelpers::UpdateShopifyProductStatus($shopName, $shopAccessToken, $userId, 'ACTIVE', 'ARCHIVED', $unSelectedRangeArray);
                    }
                    
                    $orderSettingData = App\Http\Helpers\CustomHelpers::getOrderSettingData($userData->shopify_auth_session_id);  
                    
                    sleep(10);
                    
                    if(!empty($orderSettingData)){
                        $responseData = [
                            'status' => true,
                            'data' => $orderSettingData,
                            'message' => 'Order setting updated successfully'
                        ];
                    }else{
                        $responseData = [
                            'status' => false,
                            'data' => [],
                            'message' => 'Order setting information not found'
                        ];
                    }  
                }else{
                    $responseData = [
                        'status' => false,
                        'data' => [],
                        'message' => $marginErrorMessage
                    ];
                }
                
            }else{
                $responseData = [
                    'status' => false,
                    'data' => [],
                    'message' => 'Please select any product range from order setting page'
                ];
            }
        }else{
            $responseData = [
                'status' => false,
                'data' => [],
                'message' => 'Sync is currently running, please try again shortly.'
            ];  
        }
        
    }else{
        $responseData = [
            'status' => false,
            'data' => [],
            'message' => 'User not found, First Please submit username and api key on the setting tab'
        ];
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::get('/api/get-how-to-page-contents', function (Request $request) {
    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
    $shopify_auth_token = $session->getAccessToken();
    
    $sessionInfo = Session::select('theme_status')->where('shop', $session->getShop())->where('access_token', $shopify_auth_token)->first();
    if(!empty($sessionInfo) && $sessionInfo->theme_status == 1){

        $userInfo = User::where('id', 1)->first();
        $howToPageTitle = '';
        $howToPageVideoLink = '';
        if(!empty($userInfo)){
            $howToPageTitle = $userInfo->how_to_page_title;
            $howToPageVideoLink = $userInfo->how_to_page_video_link;
        }
        $pageContentData = PersonalisedGiftAppHowToPageContent::get();
        if(!empty($pageContentData) && $pageContentData->count() > 0){
            $responseData = [
                'status' => true,
                'data' => $pageContentData,
                'page_title' => $howToPageTitle,
                'video_link' => $howToPageVideoLink,
                'theme_status' => true
            ];
            
        }else{
            $responseData = [
                'status' => false,
                'data' => [],
                'page_title' => '',
                'video_link' => '',
                'theme_status' => true
            ];
        }
    }else{
        $responseData = [
            'status' => false,
            'data' => [],
            'page_title' => '',
            'video_link' => '',
            'theme_status' => false
        ];
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::get('/api/get-faq-page-contents', function (Request $request) {
    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
    $shopify_auth_token = $session->getAccessToken();
    
    $sessionInfo = Session::select('theme_status')->where('shop', $session->getShop())->where('access_token', $shopify_auth_token)->first();
    if(!empty($sessionInfo) && $sessionInfo->theme_status == 1){

        $userInfo = User::where('id', 1)->first();
        $faqPageTitle = '';
        if(!empty($userInfo)){
            $faqPageTitle = $userInfo->faq_page_title;
        }

        $faqPageContentData = PersonalisedGiftAppFaqPageContent::get();
        if(!empty($faqPageContentData) && $faqPageContentData->count() > 0){
            $responseData = [
                'status' => true,
                'data' => $faqPageContentData,
                'page_title' => $faqPageTitle,
                'theme_status' => true
            ];
            
        }else{
            $responseData = [
                'status' => false,
                'data' => [],
                'page_title' => '',
                'theme_status' => true
            ];
        }
    }else{
        $responseData = [
            'status' => false,
            'data' => [],
            'page_title' => '',
            'theme_status' => false
        ];
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::get('/api/get-activate-theme-status', function (Request $request) {
    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
    $shopify_auth_token = $session->getAccessToken();

    // check theme if supports is 2.0
    $checkTheme = App\Http\Helpers\CustomHelpers::checkShopifyThemeSupport($session->getShop(), $session->getAccessToken()); 
    $veriyThemeStatus = 0;
    if(!empty($checkTheme) && count($checkTheme) > 0){
        $veriyThemeStatus = 1;
    }

    $sessionInfo = Session::where('shop', $session->getShop())->where('access_token', $shopify_auth_token)->first();
    if(!empty($sessionInfo)){
        $sessionInfo->update(['theme_status' => $veriyThemeStatus]);
    }

    $sessionDataInfo = Session::where('shop', $session->getShop())->where('access_token', $shopify_auth_token)->first();
    $themeStatus = false;
    $themeText = 'inactive';
    if(!empty($sessionDataInfo)){
        if($sessionDataInfo->theme_status == 1){
            $themeStatus = true;
            $themeText = 'active';
        }
    }
    
    $responseData = [
        'status' => $themeStatus,
        'data' => array($themeText),
        'message' => $themeText
    ];

    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::get('/api/get-categories-data', function (Request $request) {
    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
    $shopify_auth_token = $session->getAccessToken();

    $userData = PersonalisedGiftAppUser::where('shopify_auth_token', $shopify_auth_token)->first();
    if(!empty($userData)){
        
        $sendCategoryArray = [];
        $categoryProductCount = new PersonalisedGiftAppCategoryProduct();
        $categoryData = PersonalisedGiftAppCategory::where('personalised_gift_app_user_id', $userData->id)->orderBy('name', 'asc')->get();
        if(!empty($categoryData) && $categoryData->count() > 0){
                
            foreach($categoryData as $key => $category){
                
                $categoryPCount = $categoryProductCount->getCategoryProductCount($category->id);
                $category_label = '';
                $category_value = '';
                
                if($categoryPCount > 0){
                    if($key == 0){
                        $category_label = 'Select Category';
                        $category_value = '';
                    }else{
                        $category_label = $category->name.' ('.$category->slug.')';
                        $category_value = $category->slug;
                    }
                }
        
                if(!empty($category_label) || !empty($category_value)){
                    $sendCategoryArray[$key]['label'] = $category_label;
                    $sendCategoryArray[$key]['value'] = $category_value;
                }
            }
        }
        
        $responseData = [
            'status' => true,
            'data' => array_values($sendCategoryArray),
            'theme_status' => true
        ];
        
    }else{
        $responseData = [
            'status' => true,
            'data' => [],
            'theme_status' => true
        ];
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::get('/api/get-products-data', function (Request $request) {

    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
    $shopify_auth_token = $session->getAccessToken();

    $checkSession = Session::where('access_token', $shopify_auth_token)->first();
    if(!empty($checkSession) && $checkSession->theme_status == 1){
        $shopData = PersonalisedGiftAppUser::where('shopify_auth_token', $shopify_auth_token)->first();
        $totalProduct = 0;
        $syncingProducts = 0;
        $pgs_app_active_status = 0;
        $pgs_app_import_product_status = 0;
        $decodeRangeType = [];
        
        if(!empty($shopData)){
            
            $totalProduct = $shopData->total_pgs_products;
            if(!empty($shopData->price_range_values)){
                
                $decodeRangeType = explode(",", $shopData->price_range_values);
                $totalProduct = PersonalisedGiftAppUserProductRanges::where('personalised_gift_app_user_id', $shopData->id)->whereIn('product_range_type', $decodeRangeType)->sum('total_products');
                
                /* $decodeRangeType = explode(",", $shopData->price_range_values);
                if(count($decodeRangeType) <= 1){
                    $totalProduct  = PersonalisedGiftAppUserProductRanges::where('personalised_gift_app_user_id', $shopData->id)->sum('total_products');
                }else{
                    $totalProduct = $shopData->total_pgs_products;
                } */
                
            }
            
            $syncingProducts = $shopData->total_sync_products;
            $pgs_app_active_status = $shopData->personalised_app_user_status;
            $pgs_app_import_product_status = $shopData->import_all;
        }
        
        if($pgs_app_active_status == 1){

            $syncingProducts = PersonalisedGiftAppProduct::whereIn('personalised_gift_app_products.listing_type', $decodeRangeType)->count();

            $categoryId = '';
            if(isset($request->category) && !empty($request->category)){
                $categoryInfo = PersonalisedGiftAppCategory::where('personalised_gift_app_user_id', $shopData->id)->where('slug', $request->category)->first();
                if(!empty($categoryInfo)){
                    $categoryId = $categoryInfo->id;
                }
            }
        
            $query = PersonalisedGiftAppProduct::join('personalised_gift_app_category_products', 'personalised_gift_app_category_products.personalised_gift_app_product_id', '=', 'personalised_gift_app_products.id')->where('personalised_gift_app_products.personalised_gift_app_user_id', $shopData->id);
            
            if(isset($categoryId) && !empty($categoryId)){
                $query = $query->where('personalised_gift_app_category_products.personalised_gift_app_category_id', $categoryId);
            }
            
            if(isset($request->search) && !empty($request->search)){
                $query = $query->where('personalised_gift_app_products.post_title', 'LIKE', '%'.$request->search.'%');
            }
            
            if(!empty($decodeRangeType) && count($decodeRangeType) > 0){
                $query = $query->whereIn('personalised_gift_app_products.listing_type', $decodeRangeType);
            }

            $pgsProducts = $query->groupBy('personalised_gift_app_products.id')->orderBy('shopify_product_id', 'desc')->paginate(50, array('personalised_gift_app_products.*'))->toArray();

            if(!empty($pgsProducts) && count($pgsProducts) > 0){

                $prev_page_url = $pgsProducts['prev_page_url'];
                $next_page_url = $pgsProducts['next_page_url'];
                if((isset($request->search) && !empty($request->search)) || (isset($request->category) && !empty($request->category))){

                    if(!empty($prev_page_url)){
                        $prev_page_url = $prev_page_url.'&search='.$request->search.'&category='.$request->category;
                    }
                    if(!empty($next_page_url)){
                        $next_page_url = $next_page_url.'&search='.$request->search.'&category='.$request->category;
                    }
                }

                $responseData = [
                    'status' => true,
                    'data' => $pgsProducts['data'],
                    'prev_page_url' => $prev_page_url,
                    'next_page_url' => $next_page_url,
                    // 'total_pgs_products' => $totalProduct,
                    'total_pgs_products' => $syncingProducts,
                    'syncing_total_pgs_products' => $syncingProducts,
                    'pgs_app_active_status' => $pgs_app_active_status,
                    'pgs_app_import_product_status' => $pgs_app_import_product_status,
                    'theme_status' => true,
                    'automated_job_status' => App\Http\Helpers\CustomHelpers::getShopRunningJobStatus($checkSession->shop),
                ];

            }else{
                $responseData = [
                    'status' => false,
                    'data' => [],
                    'theme_status' => true
                ];
            }

        }else{
            $responseData = [
                'status' => true,
                'data' => [],
                'theme_status' => true
            ];
        }
        
    }else{
        $responseData = [
            'status' => false,
            'data' => [],
            'theme_status' => false
        ];
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::post('/api/create-products-on-shopify', function (Request $request) {
    /** @var AuthSession */
    if(empty($request->product_ids)){
        $responseData = [
            'status' => false,
            'data' => [],
            'message' => 'Please select products to publish on the store'
        ];

    }else{

        $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active

        $shopify_auth_token = $session->getAccessToken();
        $checkSession = PersonalisedGiftAppUser::where('shopify_auth_token', $shopify_auth_token)->first();

        $automatedJobStatus = App\Http\Helpers\CustomHelpers::getShopRunningJobStatus($checkSession->shop_domain);
        if($automatedJobStatus == 0){

            $pgs_image_url = $_ENV['EXTRA_PGS_IMAGET_SITE_URL'];
            $productIds = explode(",", $request->product_ids);
            $decodeRanges = [];
            $persolaisedProducts = PersonalisedGiftAppProduct::whereIn('id', $productIds)->whereNull('shopify_product_id')->get();
            if(!empty($persolaisedProducts) && $persolaisedProducts->count() > 0){
                
                if(!empty($checkSession->price_range_values)){
                    $decodeRanges = explode(",", $checkSession->price_range_values);
                }

                $checkStoreMetafields = ShopifyMetafield::where('resource_type', 'store')->where('personalised_gift_app_user_id', $checkSession->id)->first();
                if(empty($checkStoreMetafields)){
                    App\Http\Helpers\CustomHelpers::createShopStoreMetafields($shopify_auth_token);    
                }

                $shopifyLocationId = '';
                $shopifySalesChannelId = '';
                if(!empty($checkSession->store_location_id)){
                    $shopifyLocationId = $checkSession->store_location_id;
                }

                if(!empty($checkSession->store_sales_channel_id)){
                    $shopifySalesChannelId = $checkSession->store_sales_channel_id;
                }

                $shopName = $checkSession->session->shop;
                $shopAccessToken = $checkSession->session->access_token;
                $shopifyAuthSessionId = $checkSession->shopify_auth_session_id;
                
                $rangeType = $checkSession->pricing_options;
                $marginPrice = $checkSession->margin_change;
                $deductPrice = $checkSession->deduct_price_change;
                $importCategoryStatus = $checkSession->import_category;
                
                $printErrors = [];
                $printRangeErrors = [];
                foreach($persolaisedProducts as $key => $persolaisedProduct){
                    
                    if(in_array($persolaisedProduct->listing_type, $decodeRanges)){
                        
                        if ($persolaisedProduct->is_publish_shopify_product == 0) {
                            
                            // create category on the shopify store
                            $productId = $persolaisedProduct->id;
                            $shopifyCollectionIds = [];
                            
                            if($importCategoryStatus == 1){
                                $shopifyCollectionIds = App\Http\Helpers\CustomHelpers::createShopifyCollectionRequest($shopName, $shopAccessToken, $productId, $shopifySalesChannelId); 
                            }
            
                            // create product on the shopify store
                            App\Http\Helpers\CustomHelpers::createShopifyProductRequest($shopName, $shopAccessToken, $productId, $shopifyLocationId, $shopifySalesChannelId, $shopifyAuthSessionId, $shopifyCollectionIds, $rangeType, $marginPrice, $deductPrice, $importCategoryStatus, $checkSession->id); 
                        
                        }else{
                            $printErrors[] = $persolaisedProduct->post_title;
                        }
                    }else{
                        $printRangeErrors[] = $persolaisedProduct->post_title;
                    }
                }
                
                if(!empty($printErrors) && count($printErrors) > 0){
                    if(count($productIds) > 1){
                        $txtmsg = "Product not created on the shopify store, other product publish on the store";
                    }else{
                        $txtmsg = "Product not created on the shopify store";
                    }
                    
                }elseif(!empty($printRangeErrors) && count($printRangeErrors) > 0){
                    $txtmsg = "Product range not added on the order setting page";
                    
                }else{
                    $txtmsg = "Products successfully created on the store";
                }

                $responseData = [
                    'status' => true,
                    'data' => [],
                    'message' => $txtmsg
                ];

            }else{
                $responseData = [
                    'status' => false,
                    'data' => [],
                    'message' => 'Products already created on the shopify store.'
                ];
            }
        }else{
            $responseData = [
                'status' => false,
                'data' => [],
                'message' => 'Sync is currently running, please try again shortly.'
            ];  
        }
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::get('/api/get-pgs-products', function (Request $request) {
    /** @var AuthSession */
    $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
    $shopify_auth_token = $session->getAccessToken();
    
    $shopInfo = PersonalisedGiftAppUser::where('shopify_auth_token', $shopify_auth_token)->first();
    if(!empty($shopInfo) && $shopInfo->personalised_app_user_status == 1){
        
        $automatedJobStatus = App\Http\Helpers\CustomHelpers::getShopRunningJobStatus($shopInfo->shop_domain);
        if($automatedJobStatus == 0){
            
            if($shopInfo->price_range_values != ''){
                
                $username = $shopInfo->username;
                $apikey = $shopInfo->api_key;  
                //$lastPage = $shopInfo->last_sync_page;  
                $shopifyAuthSessionId = $shopInfo->shopify_auth_session_id;
                
                //$shopifyLocationId = '';
                //$shopifySalesChannelId = '';
                //if(!empty($shopInfo->store_location_id)){
                //    $shopifyLocationId = $shopInfo->store_location_id;
                //}
        
                //if(!empty($shopInfo->store_sales_channel_id)){
                //    $shopifySalesChannelId = $shopInfo->store_sales_channel_id;
                ///}
                
                $shopName = $shopInfo->session->shop;
                // $shopAccessToken = $shopInfo->session->access_token;
                
                // $rangeType = $shopInfo->pricing_options;
                // $marginPrice = $shopInfo->margin_change;
                // $deductPrice = $shopInfo->deduct_price_change;
                // $userId = $shopInfo->id;
                // $importCategoryStatus = $shopInfo->import_category;
                
                App\Http\Helpers\CustomHelpers::getPgsTotalPaginationRecords($shopifyAuthSessionId, $apikey, $username); 
                
                App\Http\Helpers\CustomHelpers::syncPgsAppProducts($apikey, $username, $shopName);
                
                sleep(10);
                $message = "Product synchronization are in progress, please wait some time";
                
                $responseData = [
                    'status' => true,
                    'data' => [],
                    'message' => $message
                ];
                
            }else{
                
                $responseData = [
                    'status' => false,
                    'data' => [],
                    'message' => "Please choose a product range from order settings."
                ];
            }
        }else{
            $responseData = [
                'status' => false,
                'data' => [],
                'message' => 'Sync is currently running, please try again shortly.'
            ];  
        }
        
    }else{
        $responseData = [
            'status' => false,
            'data' => [],
            'message' => "Shop information not found / Shop is not active."
        ];
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');


Route::post('/api/create-shopify-files', function (Request $request) {
    /** @var AuthSession */
    if(empty($request->type)){
        $responseData = [
            'status' => false,
            'data' => [],
            'message' => 'Please select type'
        ];

    }else{

        $session = $request->get('shopifySession'); // Provided by the shopify.auth middleware, guaranteed to be active
        $shopify_auth_token = $session->getAccessToken();
        $shopInfo = PersonalisedGiftAppUser::where('shopify_auth_token', $shopify_auth_token)->first();
        if(!empty($shopInfo)){
            $shopName = $shopInfo->session->shop;
            $shopAccessToken = $shopInfo->session->access_token;

            $totalPgsFonts = $shopInfo->total_pgs_fonts;
            $totalPgsCliparts = $shopInfo->total_pgs_cliparts;
    
            $totalSendPgsFonts = $shopInfo->total_send_shopify_files_pgs_fonts;
            $totalSendPgsCliparts = $shopInfo->total_send_shopify_files_pgs_cliparts;
            
            if($request->type == 'fonts'){
                $records = PersonalisedGiftAppFont::where('personalised_gift_app_user_id', $shopInfo->id)->whereNull('shopify_file_id')->limit(1)->get();
                $typeFile = 'font_file';

            }else{
                $records = PersonalisedGiftAppClipart::where('personalised_gift_app_user_id', $shopInfo->id)->whereNull('shopify_file_id')->limit(1)->get();
                $typeFile = 'image_url';
            }

            if(!empty($records) && $records->count() > 0){
                foreach($records as $record){
                    $filesArray = [
                        'originalSource' => $record->$typeFile,
                        'contentType' => 'FILE',
                        'alt' => "Json File",
                    ]; // for add files
                    
                    Log::channel('shopify-files')->info("Create Shopify Files Request Data -".json_encode($filesArray). 'for type - '.$typeFile);
    
                    $productFileResponse = ProductCreator::callCreateShopifyFiles($shopName, $shopAccessToken, $filesArray);

                    Log::channel('shopify-files')->info("Create Shopify Files Response Data -".json_encode($productFileResponse). 'for type - '.$typeFile);

                    if(!empty($productFileResponse)){
                        if(!empty($productFileResponse->data)){
                            $shopifyGraphQlFileId = $productFileResponse->data->fileCreate->files[0]->id;
                        }
                        $record->update(['shopify_file_id' => $shopifyGraphQlFileId]);
                    }
                }

                App\Http\Helpers\CustomHelpers::UpdateShopifyFontClipartFile($shopInfo->shopify_auth_session_id, $request->type);    

                $responseData = [
                    'status' => true,
                    'data' => [],
                    'message' => $request->type.' shopify files created successfully',
                    'total_pgs_fonts' => $totalPgsFonts,
                    'total_pgs_cliparts' => $totalPgsCliparts,
                    'send_shopify_pgs_fonts' => $totalSendPgsFonts,
                    'send_shopify_pgs_cliparts' => $totalSendPgsCliparts,
                    'pgs_app_status' => "success",
                    'pgs_app_status_message' => "Verification Successful",
                ];
            }else{
                $responseData = [
                    'status' => false,
                    'data' => [],
                    'message' => 'No Records founds'
                ];
            }
        }
    }
    return json_encode($responseData);
    
})->middleware('shopify.auth');

Route::post('/api/customer-data-request', function (Request $request) {

    $headerData = $request->header();
    Log::channel('shopify-customer-request-webhook-request')->info('Customer Request Hook Call');
    Log::channel('shopify-customer-request-webhook-request')->info('header request data - '.json_encode($headerData));
    Log::channel('shopify-customer-request-webhook-request')->info('request data - '.json_encode($request->all()));

    if(isset($headerData['x-shopify-access-token']) && !empty($headerData['x-shopify-access-token'])){

        $checkSession = Session::where('access_token', $headerData['x-shopify-access-token'])->first();
        if(!empty($checkSession)){

            App\Http\Helpers\CustomHelpers::storeShopifyMandatoryWebhookRequest('customer-request', $request->all()); 
            return response()->json(['status' => true], 200);
        }else{
            return response()->json(['status' => false], 401);
        }  

    }else{
        return response()->json(['status' => false], 401);
    }
});

Route::post('/api/customer-redact-request', function (Request $request) {

    $headerData = $request->header();
    Log::channel('shopify-customer-redact-webhook-request')->info('Customer Redact Request Hook Call');
    Log::channel('shopify-customer-redact-webhook-request')->info('header request data - '.json_encode($headerData));
    Log::channel('shopify-customer-redact-webhook-request')->info('request data - '.json_encode($request->all()));

    if(isset($headerData['x-shopify-access-token']) && !empty($headerData['x-shopify-access-token'])){

        $checkSession = Session::where('access_token', $headerData['x-shopify-access-token'])->first();
        if(!empty($checkSession)){
            
            App\Http\Helpers\CustomHelpers::storeShopifyMandatoryWebhookRequest('customer-redact-request', $request->all()); 
            return response()->json(['status' => true], 200);
        }else{
            return response()->json(['status' => false], 401);
        }       
    }else{
        return response()->json(['status' => false], 401);
    }
});

Route::post('/api/shop-redact-request', function (Request $request) {

    $headerData = $request->header();
    Log::channel('shopify-shop-redact-webhook-request')->info('Shop Redact Request Hook Call');
    Log::channel('shopify-shop-redact-webhook-request')->info('header request data - '.json_encode($headerData));
    Log::channel('shopify-shop-redact-webhook-request')->info('request data - '.json_encode($request->all()));

    if(isset($headerData['x-shopify-access-token']) && !empty($headerData['x-shopify-access-token'])){

        $checkSession = Session::where('access_token', $headerData['x-shopify-access-token'])->first();
        if(!empty($checkSession)){

            App\Http\Helpers\CustomHelpers::storeShopifyMandatoryWebhookRequest('shop-redact-request', $request->all()); 
            return response()->json(['status' => true], 200);
        }else{
            return response()->json(['status' => false], 401);
        }  

    }else{
        return response()->json(['status' => false], 401);
    }
});

Route::get('/register', function () {
    return redirect('login');
});

Route::get('/home', function (Request $request) {
    if(!empty(Auth::user())){
        return redirect('admin/dashboard');
    }else{
        return redirect('login');
    }
});

Route::get('check-pgs-product-images', [App\Http\Controllers\FrontController::class, 'checkPgsProductImages'])->name('check-pgs-product-images');

Route::get('create-shopify-font-files', [App\Http\Controllers\FrontController::class, 'createShopifyFontFiles'])->name('create-shopify-font-files');

Route::get('user-verify-email/{email}/{user_id}', [App\Http\Controllers\FrontController::class, 'verifyUserEmail'])->name('user-verify-email');

Route::get('createhookorders', [App\Http\Controllers\CronController::class, 'cronJobForCreateHookOrders'])->name('createhookorders');

Route::get('deleted-hook-products', [App\Http\Controllers\CronController::class, 'CronJobForDeletedHookProducts'])->name('deleted-hook-products');

Route::get('create-pgs-orders', [App\Http\Controllers\CronController::class, 'CronJobForCreatePgsOrders'])->name('create-pgs-orders');

Route::get('pgs-order-status', [App\Http\Controllers\CronController::class, 'getPgsOrderStatusInformation'])->name('pgs-order-status');

Route::get('sync-pgs-product-quantity', [App\Http\Controllers\CronController::class, 'syncPgsProductQunatity'])->name('sync-pgs-product-quantity');

Route::get('update-modify-pgs-product-webhook-status-requests', [App\Http\Controllers\CronController::class, 'updateModifyPgsWebhookRequest'])->name('update-modify-pgs-product-webhook-status-requests');

Route::get('update-deleted-pgs-product-webhook-status-requests', [App\Http\Controllers\CronController::class, 'updateDeletedPgsWebhookRequest'])->name('update-deleted-pgs-product-webhook-status-requests');

Route::get('update-dropship-price', [App\Http\Controllers\CronController::class, 'updateCustomDropShipPrice'])->name('update-dropship-price');

Route::get('accept-fullfillment-request-status', [App\Http\Controllers\CronController::class, 'acceptedOrderFulfillmentRequest'])->name('accept-fullfillment-request-status');

// Route::get('create-pgs-product-auomated-jobs', [App\Http\Controllers\CronController::class, 'createPgsProductAutomatedJob'])->name('create-pgs-product-auomated-jobs');

// Route::get('check-duplicate-products', [App\Http\Controllers\FrontController::class, 'checkDuplicateProducts'])->name('check-duplicate-products');

Route::get('logout', [App\Http\Controllers\Auth\LoginController::class, 'logout']);

Route::get('fonts/{fontfile}', function ($fontfile) {
    return response()->file(public_path('pgs-app/fonts/'.$fontfile));
})->middleware('cors');

Route::get('option-fonts/{fontfile}', function ($fontfile) {
    return response()->file(public_path('pgs-app/option-fonts/'.$fontfile));
})->middleware('cors');

Route::get('user-fonts/{fontfile}', function ($fontfile) {
    return response()->file(public_path('pgs-app/user-fonts/'.$fontfile));
})->middleware('cors');

Route::get('cliparts/{clipartfile}', function ($clipartfile) {
    return response()->file(public_path('pgs-app/cliparts/'.$clipartfile));
})->middleware('cors');

Route::get('pgs-app/files/{file}', function ($file, $type) {
    return response()->file(public_path('pgs-app/files/'.$file));
})->middleware('cors');

Route::post('order-fullfillment-request', function(Request $request){
   Log::channel('shopify-order-fulfillment-request')->info("send order fulfillment request data -".json_encode($request->all())); 
});

Route::post('order-fullfillment-request/fulfillment_order_notification', function(Request $request){
   Log::channel('shopify-order-fulfillment-request')->info("fulfillment order notification request data - -".json_encode($request->all())); 
});

Route::get('order-fullfillment-request/fetch_stock', function(Request $request){
   Log::channel('shopify-order-fulfillment-request')->info("fulfillment order fetch stock request data - -".json_encode($request->all())); 
});

Auth::routes();

include ('admin.php');
