<?php

declare(strict_types=1);

namespace App\Lib;

use App\Exceptions\ShopifyProductCreatorException;
use Shopify\Auth\Session;
use Shopify\Clients\Graphql;

class ProductCreator
{

    private const GET_SHOPIFY_SHOP = <<<'QUERY'
    query {
        shop {
            id
            name
            url
        }
    }
    QUERY;

    private const GET_STORE_SHOPIFY_COLLECTION = <<<'QUERY'
    query GetShopifyCollectionByQuery($query: String!){
        collections(first:1,query: $query) {
            edges {
                node {
                    id
                    title
                    handle
                }
            }
        }
    }
    QUERY;

    private const GET_STORE_SHOPIFY_PRODUCTS = <<<'QUERY'
    query GetShopifyProductsByQuery($query: String!){
        products(first:1,query: $query) {
            edges {
                node {
                    id
                    title
                    handle
                }
            }
        }
    }
    QUERY;
    
    private const GET_STORE_SHOPIFY_MULTIPLE_PRODUCTS = <<<'QUERY'
    query GetShopifyProductsByQuery($query: String!){
        products(first:10,query: $query) {
            edges {
                node {
                    id
                    title
                    handle
                }
            }
        }
    }
    QUERY;

    private const GET_STORE_LOCATION = <<<'QUERY'
    query {
        locations(first:1,query: "name:'Personalised Gift App Fulfillment Service' AND address1:'CNC Group Ltd, 6 Barnes Close'") {
            edges {
                node {
                    id
                    name
                    address{
                        address1
                        city
                        country
                        phone
                        province
                        zip
                    }
                }
            }
        }
    }
    QUERY;

    private const GET_SALES_CHANNELS = <<<'QUERY'
    query {
        publications(first:1) {
            edges {
                node {
                    id
                    name
                }
            }
        }
    }
    QUERY;

    private const GET_SHOPIFY_SINGLE_PRODUCT = <<<'QUERY'
    query GetProductsById($id: ID!) {
        product(id: $id) {
            id
            title
            variants (first: 1) {
                edges {
                    node {
                        id
                        inventoryItem{
                            id
                        }
                    }
                }
            }
            metafields(first: 2) {
                edges {
                  node {
                    id
                    namespace
                    key
                    value
                  }
                }
            }
        }
    }
    QUERY;

    private const GET_SINGLE_ORDER = <<<'QUERY'
    query GetOrderById($id: ID!) {
        order(id: $id) {
            id
            name
            fulfillmentOrders (first:20) {
                nodes{
                    id
                    lineItems(first: 20){
                        edges {
                            node{
                                id
                                sku
                                productTitle
                            }
                        }
                    }
                }
            }  
            shippingAddress {
                city
                name
                address1
                address2
                country
                province
                provinceCode
                zip
                phone
            }
        }
    }
    QUERY;

    private const UPDATE_SHOP_METAFIELDS = <<<'QUERY'
    mutation metafieldsSet($metafields: [MetafieldsSetInput!]!) {
        metafieldsSet(metafields: $metafields) {
            metafields {
                id
                ownerType
                namespace
                key
                type
                value
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const CREATE_SHOPIFY_WEBHOOKS = <<<'QUERY'
    mutation webhookSubscriptionCreate($topic: WebhookSubscriptionTopic!, $webhookSubscription: WebhookSubscriptionInput!) {
        webhookSubscriptionCreate(topic: $topic, webhookSubscription: $webhookSubscription) {
            webhookSubscription {
                id
                endpoint
                format
                topic
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const CREATE_PRODUCTS_MUTATION = <<<'QUERY'
    mutation populateProduct($input: ProductInput!) {
        productCreate(input: $input) {
            product {
                id
                handle
                title
                status
                variants (first: 1) {
                    edges {
                        node {
                            id
                            inventoryItem{
                                id
                            }
                        }
                    }
                }
                metafields(first: 2) {
                    edges {
                      node {
                        id
                        namespace
                        key
                        value
                        type
                      }
                    }
                }
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const UPDATE_PRODUCTS_MUTATION = <<<'QUERY'
    mutation productUpdate($input: ProductInput!) {
        productUpdate(input: $input) {
            product {
                id
                handle
                title
                status
                variants (first: 1) {
                    edges {
                        node {
                            id
                            inventoryItem{
                                id
                            }
                        }
                    }
                }
                metafields(first: 2) {
                    edges {
                      node {
                        id
                        namespace
                        key
                        value
                        type
                      }
                    }
                }
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const UPDATE_PRODUCT_VARIANTS_MUTATION = <<<'QUERY'
    mutation productVariantUpdate($input: ProductVariantInput!) {
        productVariantUpdate(input: $input) {
            product {
                id
            }
            productVariant {
                id
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const CREATE_CATEGORY_MUTATION = <<<'QUERY'
    mutation populateCategory($input: CollectionInput!) {
        collectionCreate(input: $input) {
            collection {
                id
                handle
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const UPDATE_CATEGORY_MUTATION = <<<'QUERY'
    mutation populateCategory($input: CollectionInput!) {
        collectionUpdate(input: $input) {
            collection {
                id
                handle
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const CREATE_STORE_LOCATION = <<<'QUERY'
    mutation locationAdd($input: LocationAddInput!) {
        locationAdd(input: $input) {
            location {
                id
                name
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;
    
    private const CREATE_FULFILLMENT_SERVICE_LOCATION = <<<'QUERY'
    mutation fulfillmentServiceCreate($name: String!, $callbackUrl: URL!, $fulfillmentOrdersOptIn: Boolean!, $inventoryManagement: Boolean!, $permitsSkuSharing: Boolean!) {
        fulfillmentServiceCreate(name: $name, callbackUrl: $callbackUrl, fulfillmentOrdersOptIn: $fulfillmentOrdersOptIn, inventoryManagement: $inventoryManagement, permitsSkuSharing: $permitsSkuSharing) {
            fulfillmentService {
               	id
                serviceName
                callbackUrl
                fulfillmentOrdersOptIn
                inventoryManagement
                permitsSkuSharing
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const PUBLISH_PRODUCT_ON_SALES_CHANNELS = <<<'QUERY'
    mutation publishablePublish($id: ID!, $input: [PublicationInput!]!) {
        publishablePublish(id: $id, input: $input) {
            publishable {
                availablePublicationCount
                publicationCount
            }
            shop {
                publicationCount
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const UPDATE_SHOPIFY_INVENTORY_QUANTITY = <<<'QUERY'
    mutation ActivateInventoryItem($inventoryItemId: ID!, $locationId: ID!, $available: Int) {
        inventoryActivate(inventoryItemId: $inventoryItemId, locationId: $locationId, available: $available) {
            inventoryLevel {
                id
                available
                item {
                    id
                }
                location {
                    id
                }
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const CREATE_SHOPIFY_FILES = <<<'QUERY'
    mutation fileCreate($files: [FileCreateInput!]!) {
        fileCreate(files: $files) {
            files {
                createdAt
                fileStatus
                ... on GenericFile {
                    id
                    alt
                    url
                }
                preview{
                    image{
                        id
                        url
                    }
                    status
                }
                fileErrors{
                    code
                    details
                    message
                }
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const DELETE_SHOPIFY_METAFILED = <<<'QUERY'
    mutation metafieldDelete($input: MetafieldDeleteInput!) {
        metafieldDelete(input: $input) {
            deletedId
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const DELETE_SHOPIFY_WEBHOOKS = <<<'QUERY'
    mutation webhookSubscriptionDelete($id: ID!) {
        webhookSubscriptionDelete(id: $id) {
            deletedWebhookSubscriptionId
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const DELETE_SHOPIFY_COLLECTION = <<<'QUERY'
    mutation collectionDelete($input: CollectionDeleteInput!) {
        collectionDelete(input: $input) {
            deletedCollectionId
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const DELETE_SHOPIFY_PRODUCT = <<<'QUERY'
    mutation productDelete($input: ProductDeleteInput!) {
        productDelete(input: $input) {
            deletedProductId
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const CREATE_ORDER_FULLFILLMENT = <<<'QUERY'
    mutation fulfillmentCreateV2($fulfillment: FulfillmentV2Input!) {
        fulfillmentCreateV2(fulfillment: $fulfillment) {
            fulfillment{
                id
                status
                trackingInfo {
                    company
                    number
                    url
                }
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;
    
    private const CHANGE_PRODUCT_STATUS = <<<'QUERY'
    mutation productChangeStatus($productId: ID!, $status: ProductStatus!) {
        productChangeStatus(productId: $productId, status: $status) {
            product {
                id
                status
            }
            userErrors {
                field
                message
            }
        }
    }
    QUERY;

    private const UPDATE_INVENTORY_ITEM_MUTATION = <<<'QUERY'
    mutation inventoryItemUpdate($id: ID!, $input: InventoryItemUpdateInput!) {
        inventoryItemUpdate(id: $id, input: $input) {
            inventoryItem {
                id
                unitCost {
                    amount
                }
                tracked
                countryCodeOfOrigin
                provinceCodeOfOrigin
                harmonizedSystemCode
            }
            userErrors {
              field
              message
            }
        }
    }
    QUERY;


    public static function call(Session $session, int $count)
    {
        $client = new Graphql($session->getShop(), $session->getAccessToken());

        for ($i = 0; $i < $count; $i++) {
            $response = $client->query(
                [
                    "query" => self::CREATE_PRODUCTS_MUTATION,
                    "variables" => [
                        "input" => [
                            "title" => self::randomTitle(),
                            "variants" => [["price" => self::randomPrice()]],
                        ]
                    ]
                ],
            );

            if ($response->getStatusCode() !== 200) {
                throw new ShopifyProductCreatorException($response->getBody()->__toString(), $response);
            }
        }
    }

    public static function getShopifyShopInformation(Session $session)
    {
        $client = new Graphql($session->getShop(), $session->getAccessToken());
        $shop_response = $client->query(
            [
                "query" => self::GET_SHOPIFY_SHOP,
            ],
        );
        
        if ($shop_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($shop_response->getBody()->__toString(), $shop_response);
        }else{
            return json_decode($shop_response->getBody()->__toString());
        }
    }

    public static function createShopifyStoreWebhooks($shop, $shopToken, $webhookData)
    {
        $client = new Graphql($shop, $shopToken);
        $webhook_response = $client->query(
            [
                "query" => self::CREATE_SHOPIFY_WEBHOOKS,
                    "variables" => $webhookData
            ],
        );
        
        if ($webhook_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($webhook_response->getBody()->__toString(), $webhook_response);
        }else{
            return json_decode($webhook_response->getBody()->__toString());
        }
    }

    public static function callCreateCustomProducts($shop, $shopToken, $sendProductArray)
    {
        $client = new Graphql($shop, $shopToken);
        $create_product_response = $client->query(
            [
                "query" => self::CREATE_PRODUCTS_MUTATION,
                "variables" => [
                    "input" => $sendProductArray
                ]
            ],
        );
    
        if ($create_product_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($create_product_response->getBody()->__toString(), $create_product_response);
        }else{
            return json_decode($create_product_response->getBody()->__toString());
        }
    }

    public static function callUpdateCustomProducts($shop, $shopToken, $sendProductArray)
    {
        $client = new Graphql($shop, $shopToken);
        $update_product_response = $client->query(
            [
                "query" => self::UPDATE_PRODUCTS_MUTATION,
                "variables" => [
                    "input" => $sendProductArray
                ]
            ],
        );
    
        if ($update_product_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($update_product_response->getBody()->__toString(), $update_product_response);
        }else{
            return json_decode($update_product_response->getBody()->__toString());
        }
    }

    public static function updateProductVariantPrice($shop, $shopToken, $sendVariantArray)
    {
        $client = new Graphql($shop, $shopToken);
        $update_variant_response = $client->query(
            [
                "query" => self::UPDATE_PRODUCT_VARIANTS_MUTATION,
                "variables" => [
                    "input" => $sendVariantArray
                ]
            ],
        );
    
        if ($update_variant_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($update_variant_response->getBody()->__toString(), $update_variant_response);
        }else{
            return json_decode($update_variant_response->getBody()->__toString());
        }
    }

    public static function updateInventoryItemCostPrice($shop, $shopToken, $sendInventoryArray, $inventoryId)
    {
        $client = new Graphql($shop, $shopToken);
        $update_inventory_response = $client->query(
            [
                "query" => self::UPDATE_INVENTORY_ITEM_MUTATION,
                "variables" => [
                    "id" => $inventoryId,
                    "input" => $sendInventoryArray
                ]
            ],
        );
    
        if ($update_inventory_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($update_inventory_response->getBody()->__toString(), $update_inventory_response);
        }else{
            return json_decode($update_inventory_response->getBody()->__toString());
        }
    }

    public static function callCreateCustomCategory($shop, $shopToken, $sendCategoryArray)
    {
        $client = new Graphql($shop, $shopToken);
        $category_response = $client->query(
            [
                "query" => self::CREATE_CATEGORY_MUTATION,
                "variables" => [
                    "input" => $sendCategoryArray
                ]
            ],
        );
        
        if ($category_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($category_response->getBody()->__toString(), $category_response);
        }else{
            return json_decode($category_response->getBody()->__toString());
        }
    }

    public static function callUpdateCustomCategory(Session $session, $sendUpdateCategoryArray)
    {
        $client = new Graphql($session->getShop(), $session->getAccessToken());
        $update_category_response = $client->query(
            [
                "query" => self::UPDATE_CATEGORY_MUTATION,
                "variables" => [
                    "input" => $sendUpdateCategoryArray
                ]
            ],
        );
        
        if ($update_category_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($update_category_response->getBody()->__toString(), $update_category_response);
        }else{
            return json_decode($update_category_response->getBody()->__toString());
        }
    }

    public static function checkShopifyStoreExistCollection($shop, $shopToken, $query)
    {
        $client = new Graphql($shop, $shopToken);
        $single_collection_response = $client->query(
            [
                "query" => self::GET_STORE_SHOPIFY_COLLECTION,
                "variables" => [
                    "query" => $query
                ]
            ],
        );
        
        if ($single_collection_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($single_collection_response->getBody()->__toString(), $single_collection_response);
        }else{
            return json_decode($single_collection_response->getBody()->__toString());
        }
    }

    public static function checkShopifyStoreExistProduct($shop, $shopToken, $query)
    {
        $client = new Graphql($shop, $shopToken);
        $single_product_response = $client->query(
            [
                "query" => self::GET_STORE_SHOPIFY_PRODUCTS,
                "variables" => [
                    "query" => $query
                ]
            ],
        );
        
        if ($single_product_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($single_product_response->getBody()->__toString(), $single_product_response);
        }else{
            return json_decode($single_product_response->getBody()->__toString());
        }
    }
    
    public static function checkShopifyStoreExistDuplicateProduct($shop, $shopToken, $query)
    {
        $client = new Graphql($shop, $shopToken);
        $single_product_response = $client->query(
            [
                "query" => self::GET_STORE_SHOPIFY_MULTIPLE_PRODUCTS,
                "variables" => [
                    "query" => $query
                ]
            ],
        );
        
        if ($single_product_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($single_product_response->getBody()->__toString(), $single_product_response);
        }else{
            return json_decode($single_product_response->getBody()->__toString());
        }
    }

    public static function getStoreLocation(Session $session)
    {
        $client = new Graphql($session->getShop(), $session->getAccessToken());
        $location_response = $client->query(
            [
                "query" => self::GET_STORE_LOCATION,
            ],
        );
        
        if ($location_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($location_response->getBody()->__toString(), $location_response);
        }else{
            return json_decode($location_response->getBody()->__toString());
        }
    }

    public static function createStoreLocation($shop, $shopToken, $sendLocationdata)
    {
        $client = new Graphql($shop, $shopToken);
        $location_response = $client->query(
            [
                "query" => self::CREATE_STORE_LOCATION,
                "variables" => [
                    "input" => $sendLocationdata
                ]
            ],
        );
        
        if ($location_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($location_response->getBody()->__toString(), $location_response);
        }else{
            return json_decode($location_response->getBody()->__toString());
        }
    }
    
    public static function fulfillmentServiceCreate($shop, $shopToken, $servicename, $callbackUrl)
    {
        $client = new Graphql($shop, $shopToken);
        $service_response = $client->query(
            [
                "query" => self::CREATE_FULFILLMENT_SERVICE_LOCATION,
                "variables" => [
                    "name" => $servicename,
                    "callbackUrl" => $callbackUrl,
                    "fulfillmentOrdersOptIn" => true,
                    "inventoryManagement" => true,
                    "permitsSkuSharing" => true
                ]
            ],
        );
        
        if ($service_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($service_response->getBody()->__toString(), $service_response);
        }else{
            return json_decode($service_response->getBody()->__toString());
        }
    }

    public static function getAvailableSalesChannels(Session $session)
    {
        $client = new Graphql($session->getShop(), $session->getAccessToken());
        $channel_response = $client->query(
            [
                "query" => self::GET_SALES_CHANNELS,
            ],
        );
        
        if ($channel_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($channel_response->getBody()->__toString(), $channel_response);
        }else{
            return json_decode($channel_response->getBody()->__toString());
        }
    }

    public static function callPublishProductSalesChanel($shop, $shopToken, $sendSalesChannelsData, $shopifyID)
    {
        $client = new Graphql($shop, $shopToken);
        $sales_channels_response = $client->query(
            [
                "query" => self::PUBLISH_PRODUCT_ON_SALES_CHANNELS,
                "variables" => [
                    "id" => $shopifyID,
                    "input" => $sendSalesChannelsData
                ]
            ],
        );
        
        if ($sales_channels_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($sales_channels_response->getBody()->__toString(), $sales_channels_response);
        }else{
            return json_decode($sales_channels_response->getBody()->__toString());
        }
    }

    public static function getShopifySingleProduct(Session $session, $shopifyProductID)
    {
        $client = new Graphql($session->getShop(), $session->getAccessToken());
        $channel_response = $client->query(
            [
                "query" => self::GET_SHOPIFY_SINGLE_PRODUCT,
                "variables" => [
                    "id" => $shopifyProductID
                ]
            ],
        );
        
        if ($channel_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($channel_response->getBody()->__toString(), $channel_response);
        }else{
            return json_decode($channel_response->getBody()->__toString());
        }
    }

    public static function getShopifyOrderFulfillment($shop, $shopToken, $shopifyOrderID)
    {
        $client = new Graphql($shop, $shopToken);
        $order_fulfill_response = $client->query(
            [
                "query" => self::GET_SINGLE_ORDER,
                "variables" => [
                    "id" => $shopifyOrderID
                ]
            ],
        );

        if ($order_fulfill_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($order_fulfill_response->getBody()->__toString(), $order_fulfill_response);
        }else{
            return json_decode($order_fulfill_response->getBody()->__toString());
        }
    }

    public static function updateShopMetafields($shop, $shopToken, $sendMetafieldArray)
    {
        $client = new Graphql($shop, $shopToken);
        $shop_metafield_response = $client->query(
            [
                "query" => self::UPDATE_SHOP_METAFIELDS,
                "variables" => [
                    "metafields" => $sendMetafieldArray
                ]
            ],
        );
    
        if ($shop_metafield_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($shop_metafield_response->getBody()->__toString(), $shop_metafield_response);
        }else{
            return json_decode($shop_metafield_response->getBody()->__toString());
        }
    }

    public static function updateVariantInventoryQuantity($shop, $shopToken, $availableQty, $shopifyLocationID, $shopifyInvntoryItemId)
    {
        $client = new Graphql($shop, $shopToken);
        $variant_inventory_response = $client->query(
            [
                "query" => self::UPDATE_SHOPIFY_INVENTORY_QUANTITY,
                "variables" => [
                    "available" => (int) $availableQty,
                    "inventoryItemId" => $shopifyInvntoryItemId,
                    "locationId" => $shopifyLocationID
                ]
            ],
        );
        
        if ($variant_inventory_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($variant_inventory_response->getBody()->__toString(), $variant_inventory_response);
        }else{
            return json_decode($variant_inventory_response->getBody()->__toString());
        }
    }

    public static function callCreateShopifyFiles($shop, $shopToken, $sendFileArray)
    {
        $client = new Graphql($shop, $shopToken);
        $file_response = $client->query(
            [
                "query" => self::CREATE_SHOPIFY_FILES,
                "variables" => [
                    "files" => $sendFileArray
                ]
            ],
        );
    
        if ($file_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($file_response->getBody()->__toString(), $file_response);
        }else{
            return json_decode($file_response->getBody()->__toString());
        }
    }

    public static function callDeleteMetafield($shop, $shopToken, $metafieldId)
    {
        $client = new Graphql($shop, $shopToken);
        $metafield_response = $client->query(
            [
                "query" => self::DELETE_SHOPIFY_METAFILED,
                "variables" => [
                    "input" => $metafieldId
                ]
            ],
        );
    
        if ($metafield_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($metafield_response->getBody()->__toString(), $metafield_response);
        }else{
            return json_decode($metafield_response->getBody()->__toString());
        }
    }

    public static function callDeleteShopifyWebhook($shop, $shopToken, $hookId)
    {
        $client = new Graphql($shop, $shopToken);
        $hook_response = $client->query(
            [
                "query" => self::DELETE_SHOPIFY_WEBHOOKS,
                "variables" => $hookId
            ],
        );
    
        if ($hook_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($hook_response->getBody()->__toString(), $hook_response);
        }else{
            return json_decode($hook_response->getBody()->__toString());
        }
    }

    public static function callDeleteShopifyCollection($shop, $shopToken, $collectionID)
    {
        $client = new Graphql($shop, $shopToken);
        $collection_response = $client->query(
            [
                "query" => self::DELETE_SHOPIFY_COLLECTION,
                "variables" => $collectionID
            ],
        );
    
        if ($collection_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($collection_response->getBody()->__toString(), $collection_response);
        }else{
            return json_decode($collection_response->getBody()->__toString());
        }
    }

    public static function callDeleteShopifyProduct($shop, $shopToken, $productID)
    {
        $client = new Graphql($shop, $shopToken);
        $product_response = $client->query(
            [
                "query" => self::DELETE_SHOPIFY_PRODUCT,
                "variables" => $productID
            ],
        );
    
        if ($product_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($product_response->getBody()->__toString(), $product_response);
        }else{
            return json_decode($product_response->getBody()->__toString());
        }
    }

    public static function createShopifyOrderFulfillment($shop, $shopToken, $fulfillmentData)
    {
        $client = new Graphql($shop, $shopToken);
        $order_fullfillment_response = $client->query(
            [
                "query" => self::CREATE_ORDER_FULLFILLMENT,
                "variables" => [
                    "fulfillment" => $fulfillmentData,
                    "message" => "PGS app order fulfillment"
                ]
            ],
        );
    
        if ($order_fullfillment_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($order_fullfillment_response->getBody()->__toString(), $order_fullfillment_response);
        }else{
            return json_decode($order_fullfillment_response->getBody()->__toString());
        }
    }
    
    public static function changeShopifyProductStatus($shop, $shopToken, $shopifyProductId, $status)
    {
        $client = new Graphql($shop, $shopToken);
        $product_status_response = $client->query(
            [
                "query" => self::CHANGE_PRODUCT_STATUS,
                "variables" => [
                    "productId" => $shopifyProductId,
                    "status" => $status
                ]
            ],
        );
    
        if ($product_status_response->getStatusCode() !== 200) {
            throw new ShopifyProductCreatorException($product_status_response->getBody()->__toString(), $product_status_response);
        }else{
            return json_decode($product_status_response->getBody()->__toString());
        }
    }

    private static function randomTitle()
    {
        $adjective = self::ADJECTIVES[mt_rand(0, count(self::ADJECTIVES) - 1)];
        $noun = self::NOUNS[mt_rand(0, count(self::NOUNS) - 1)];

        return "$adjective $noun";
    }

    private static function randomPrice()
    {

        return (100.0 + mt_rand(0, 1000)) / 100;
    }

    private const ADJECTIVES = [
        "autumn",
        "hidden",
        "bitter",
        "misty",
        "silent",
        "empty",
        "dry",
        "dark",
        "summer",
        "icy",
        "delicate",
        "quiet",
        "white",
        "cool",
        "spring",
        "winter",
        "patient",
        "twilight",
        "dawn",
        "crimson",
        "wispy",
        "weathered",
        "blue",
        "billowing",
        "broken",
        "cold",
        "damp",
        "falling",
        "frosty",
        "green",
        "long",
    ];

    private const NOUNS = [
        "waterfall",
        "river",
        "breeze",
        "moon",
        "rain",
        "wind",
        "sea",
        "morning",
        "snow",
        "lake",
        "sunset",
        "pine",
        "shadow",
        "leaf",
        "dawn",
        "glitter",
        "forest",
        "hill",
        "cloud",
        "meadow",
        "sun",
        "glade",
        "bird",
        "brook",
        "butterfly",
        "bush",
        "dew",
        "dust",
        "field",
        "fire",
        "flower",
    ];
}
