"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getConfigs = void 0;
const js_yaml_1 = require("js-yaml");
const fs_1 = require("fs");
const REQUIRED_CONFIGS = {
    development: {
        build_dir: true,
        entries: { main: true },
    },
};
const RESERVE_PATHS = flattenPaths({
    development: {
        build: { env: true },
        develop: { env: true },
    },
});
function getConfigs() {
    const stdin = 0;
    try {
        const configs = (0, js_yaml_1.load)((0, fs_1.readFileSync)(stdin, 'utf8'));
        if (!isValidConfigs(configs, REQUIRED_CONFIGS)) {
            throw new Error('Invalid configuration');
        }
        return jsonConfigs(configs);
        // eslint-disable-next-line no-catch-all/no-catch-all
    }
    catch (error) {
        // eslint-disable-next-line no-console
        console.log(`Failed with error: ${error}`);
        process.exit(1);
    }
}
exports.getConfigs = getConfigs;
function toCamelCase(str) {
    return str.replace(/_./g, (substr) => substr.toUpperCase()[1]);
}
function isValidConfigs(configs, requiredConfigs, paths = []) {
    Object.keys(requiredConfigs).forEach((key) => {
        const isRequired = requiredConfigs[key] !== false;
        const value = configs[key];
        if ((value === undefined || value === null) && isRequired) {
            throw new Error(`Invalid configuration. Missing \`${paths.concat(key).join('.')}\``);
        }
        if (!Array.isArray(value) && typeof value === 'object') {
            isValidConfigs(value, requiredConfigs[key], paths.concat(key));
        }
    }, {});
    return true;
}
function jsonConfigs(configs, paths = [], formatter = toCamelCase) {
    return Object.keys(configs).reduce((acc, key) => {
        const shouldReserveKey = RESERVE_PATHS.includes(paths.concat(key).join('.'));
        const formattedKey = formatter(key);
        const value = configs[key];
        if (Array.isArray(value) || typeof value !== 'object') {
            return Object.assign(Object.assign({}, acc), { [formattedKey]: configs[key] });
        }
        return Object.assign(Object.assign({}, acc), { [formattedKey]: jsonConfigs(value, paths.concat(key), shouldReserveKey ? (key) => key : formatter) });
    }, {});
}
function flattenPaths(config, paths = []) {
    return Object.keys(config).reduce((acc, key) => {
        const value = config[key];
        if (!value) {
            return acc;
        }
        if (typeof value === 'object') {
            return [...acc, ...flattenPaths(value, paths.concat(key))];
        }
        return [...acc, paths.concat(key).join('.')];
    }, []);
}
