import { Flags } from '@oclif/core';
function setupEnvironmentVariables(options) {
    /**
     * By setting DEBUG=* when --verbose is passed we are increasing the
     * verbosity of oclif. Oclif uses debug (https://www.npmjs.com/package/debug)
     * for logging, and it's configured through the DEBUG= environment variable.
     */
    if (process.argv.includes('--verbose')) {
        process.env.DEBUG = process.env.DEBUG ?? '*';
    }
    if (options.development) {
        process.env.SHOPIFY_CLI_ENV = process.env.SHOPIFY_CLI_ENV ?? 'development';
    }
}
/**
 * A function that abstracts away setting up the environment and running
 * a CLI
 * @param options - Options.
 */
export async function runCLI(options) {
    setupEnvironmentVariables(options);
    /**
     * These imports need to be dynamic because if they are static
     * they are loaded before we set the DEBUG=* environment variable
     * and therefore it has no effect.
     */
    const { errorHandler } = await import('./error-handler.js');
    const { isDevelopment } = await import('./context/local.js');
    const { run, settings, flush } = await import('@oclif/core');
    if (isDevelopment()) {
        settings.debug = true;
    }
    run(undefined, options.moduleURL)
        .then(() => flush())
        .catch(errorHandler);
}
/**
 * A function for create-x CLIs that automatically runs the "init" command.
 */
export async function runCreateCLI(options) {
    setupEnvironmentVariables(options);
    const { findUpAndReadPackageJson } = await import('./node-package-manager.js');
    const { moduleDirectory } = await import('./path.js');
    const packageJson = await findUpAndReadPackageJson(moduleDirectory(options.moduleURL));
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const packageName = packageJson.content.name;
    const name = packageName.replace('@shopify/create-', '');
    const initIndex = process.argv.findIndex((arg) => arg.includes('init'));
    if (initIndex === -1) {
        const initIndex = process.argv.findIndex((arg) => arg.match(new RegExp(`bin(\\/|\\\\)+(create-${name}|dev|run)`))) + 1;
        process.argv.splice(initIndex, 0, 'init');
    }
    await runCLI(options);
}
export async function useLocalCLIIfDetected(filepath) {
    const { isTruthy } = await import('../../private/node/context/utilities.js');
    const { environmentVariables } = await import('../../private/node/constants.js');
    const { joinPath: join } = await import('./path.js');
    const { exec } = await import('./system.js');
    // Temporary flag while we test out this feature and ensure it won't break anything!
    if (!isTruthy(process.env[environmentVariables.enableCliRedirect]))
        return false;
    // Setting an env variable in the child process prevents accidental recursion.
    if (isTruthy(process.env[environmentVariables.skipCliRedirect]))
        return false;
    // If already running via package manager, we can assume it's running correctly already.
    if (process.env.npm_config_user_agent)
        return false;
    const cliPackage = await localCliPackage();
    if (!cliPackage)
        return false;
    const correctExecutablePath = join(cliPackage.path, cliPackage.bin.shopify);
    if (correctExecutablePath === filepath)
        return false;
    try {
        await exec(correctExecutablePath, process.argv.slice(2, process.argv.length), {
            stdio: 'inherit',
            env: { [environmentVariables.skipCliRedirect]: '1' },
        });
        // eslint-disable-next-line no-catch-all/no-catch-all, @typescript-eslint/no-explicit-any
    }
    catch (processError) {
        process.exit(processError.exitCode);
    }
    return true;
}
export async function localCliPackage() {
    const { captureOutput } = await import('./system.js');
    let npmListOutput = '';
    let localShopifyCLI = {};
    try {
        npmListOutput = await captureOutput('npm', ['list', '@shopify/cli', '--json', '-l']);
        localShopifyCLI = JSON.parse(npmListOutput);
        // eslint-disable-next-line no-catch-all/no-catch-all
    }
    catch (err) {
        return;
    }
    const dependenciesList = {
        ...localShopifyCLI.peerDependencies,
        ...localShopifyCLI.devDependencies,
        ...localShopifyCLI.dependencies,
    };
    return dependenciesList['@shopify/cli'];
}
/**
 * An object that contains the flags that
 * are shared across all the commands.
 */
export const globalFlags = {
    environment: Flags.string({
        hidden: true,
        description: 'The environment to apply to the current command.',
        env: 'SHOPIFY_FLAG_ENVIRONMENT',
    }),
    verbose: Flags.boolean({
        hidden: false,
        description: 'Increase the verbosity of the logs.',
        env: 'SHOPIFY_FLAG_VERBOSE',
    }),
};
export default runCLI;
//# sourceMappingURL=cli.js.map