import { isEqual } from '../../../../public/common/lang.js';
import { groupBy, partition } from '../../../../public/common/collection.js';
import { mapValues } from '../../../../public/common/object.js';
import React, { useState, useEffect, useRef, useCallback } from 'react';
import { Box, useInput, Text } from 'ink';
import { debounce } from '@shopify/cli-kit/common/function';
import chalk from 'chalk';
import figures from 'figures';
function highlightedLabel(label, term) {
    if (!term) {
        return label;
    }
    const regex = new RegExp(term, 'i');
    return label.replace(regex, (match) => {
        return chalk.bold(match);
    });
}
function groupItems(items) {
    let index = 0;
    const [withGroup, withoutGroup] = partition(items, 'group');
    const withGroupMapped = mapValues(groupBy(withGroup, 'group'), (groupItems) => groupItems.map((groupItem) => {
        const item = { ...groupItem, key: groupItem.key ?? (index + 1).toString(), index };
        index += 1;
        return item;
    }));
    const withoutGroupMapped = withoutGroup.map((item) => {
        const newItem = { ...item, key: item.key ?? (index + 1).toString(), index };
        index += 1;
        return newItem;
    });
    return [withGroupMapped, withoutGroupMapped];
}
// eslint-disable-next-line react/function-component-definition
function SelectItemsGroup({ title, items, selectedIndex, hasMarginTop, enableShortcuts, highlightedTerm, }) {
    return (React.createElement(Box, { key: title, flexDirection: "column", marginTop: hasMarginTop ? 1 : 0 },
        title ? (React.createElement(Box, { marginLeft: 3 },
            React.createElement(Text, { bold: true }, title))) : null,
        items.map((item) => {
            const isSelected = item.index === selectedIndex;
            const label = highlightedLabel(item.label, highlightedTerm);
            return (React.createElement(Box, { key: item.key },
                React.createElement(Box, { marginRight: 2 }, isSelected ? React.createElement(Text, { color: "cyan" }, `>`) : React.createElement(Text, null, " ")),
                React.createElement(Text, { color: isSelected ? 'cyan' : undefined }, enableShortcuts ? `(${item.key}) ${label}` : label)));
        })));
}
// eslint-disable-next-line react/function-component-definition
function SelectInput({ items, onChange, enableShortcuts = true, focus = true, emptyMessage = 'No items to select.', defaultValue, highlightedTerm, loading = false, errorMessage, hasMorePages = false, morePagesMessage, infoMessage, }) {
    const defaultValueIndex = defaultValue ? items.findIndex((item) => item.value === defaultValue.value) : -1;
    const initialIndex = defaultValueIndex === -1 ? 0 : defaultValueIndex;
    const inputStack = useRef(null);
    const [selectedIndex, setSelectedIndex] = useState(initialIndex);
    const [groupedItems, ungroupedItems] = groupItems(items);
    const groupedItemsValues = [...Object.values(groupedItems).flat(), ...ungroupedItems];
    const keys = groupedItemsValues.map((item) => item.key);
    const groupTitles = Object.keys(groupedItems);
    const previousItems = useRef(items);
    const changeSelection = useCallback(({ index, usedShortcut = false }) => {
        const groupedItem = groupedItemsValues.find((item) => item.index === index);
        setSelectedIndex(index);
        onChange({
            item: items.find((item) => item.value === groupedItem.value),
            usedShortcut,
        });
    }, [items]);
    useEffect(() => {
        if (items.length === 0) {
            // reset selection when items are empty
            onChange({
                item: undefined,
                usedShortcut: false,
            });
        }
        else if (
        // reset index when items change
        !isEqual(previousItems.current.map((item) => item.value), items.map((item) => item.value))) {
            changeSelection({ index: 0 });
        }
        previousItems.current = items;
    }, [items]);
    const handleArrows = useCallback((key) => {
        const lastIndex = items.length - 1;
        if (key.upArrow) {
            changeSelection({ index: selectedIndex === 0 ? lastIndex : selectedIndex - 1 });
        }
        else if (key.downArrow) {
            changeSelection({ index: selectedIndex === lastIndex ? 0 : selectedIndex + 1 });
        }
    }, [selectedIndex, items]);
    const handleShortcuts = useCallback((input) => {
        if (keys.includes(input)) {
            const groupedItem = groupedItemsValues.find((item) => item.key === input);
            if (groupedItem !== undefined) {
                changeSelection({ index: groupedItem.index, usedShortcut: true });
            }
        }
    }, [items]);
    const debounceHandleShortcuts = useCallback(debounce((newInputStack) => {
        handleShortcuts(newInputStack);
        inputStack.current = null;
    }, 300), []);
    useInput((input, key) => {
        // check that no special modifier (shift, control, etc.) is being pressed
        if (enableShortcuts && input.length > 0 && Object.values(key).every((value) => value === false)) {
            const newInputStack = inputStack.current === null ? input : inputStack.current + input;
            inputStack.current = newInputStack;
            debounceHandleShortcuts(newInputStack);
        }
        else {
            debounceHandleShortcuts.cancel();
            inputStack.current = null;
            handleArrows(key);
        }
    }, { isActive: focus });
    const ungroupedItemsTitle = groupTitles.length > 0 ? 'Other' : undefined;
    if (loading) {
        return (React.createElement(Box, { marginLeft: 3 },
            React.createElement(Text, { dimColor: true }, "Loading...")));
    }
    else if (errorMessage && errorMessage.length > 0) {
        return (React.createElement(Box, { marginLeft: 3 },
            React.createElement(Text, { color: "red" }, errorMessage)));
    }
    else if (items.length === 0) {
        return (React.createElement(Box, { marginLeft: 3 },
            React.createElement(Text, { dimColor: true }, emptyMessage)));
    }
    else {
        return (React.createElement(Box, { flexDirection: "column" },
            groupTitles.map((title, index) => (React.createElement(SelectItemsGroup, { title: title, selectedIndex: selectedIndex, items: groupedItems[title], key: title, hasMarginTop: index !== 0, enableShortcuts: enableShortcuts, highlightedTerm: highlightedTerm }))),
            ungroupedItems.length > 0 && (React.createElement(SelectItemsGroup, { title: ungroupedItemsTitle, selectedIndex: selectedIndex, items: ungroupedItems, hasMarginTop: groupTitles.length > 0, enableShortcuts: enableShortcuts, highlightedTerm: highlightedTerm })),
            React.createElement(Box, { marginTop: 1, marginLeft: 3, flexDirection: "column" },
                hasMorePages ? (React.createElement(Text, null,
                    React.createElement(Text, { bold: true },
                        "1-",
                        items.length,
                        " of many"),
                    morePagesMessage ? `  ${morePagesMessage}` : null)) : null,
                React.createElement(Text, { dimColor: true }, infoMessage
                    ? infoMessage
                    : `Press ${figures.arrowUp}${figures.arrowDown} arrows to select, enter to confirm`))));
    }
}
export { SelectInput };
//# sourceMappingURL=SelectInput.js.map