import { schema } from '../../../public/node/schema.js';
/**
 * The schema represents an Identity token.
 */
declare const IdentityTokenSchema: schema.ZodObject<{
    accessToken: schema.ZodString;
    refreshToken: schema.ZodString;
    expiresAt: schema.ZodEffects<schema.ZodDate, Date, Date>;
    scopes: schema.ZodArray<schema.ZodString, "many">;
}, "strip", schema.ZodTypeAny, {
    accessToken: string;
    refreshToken: string;
    expiresAt: Date;
    scopes: string[];
}, {
    accessToken: string;
    refreshToken: string;
    expiresAt: Date;
    scopes: string[];
}>;
/**
 * The schema represents an application token.
 */
declare const ApplicationTokenSchema: schema.ZodObject<{
    accessToken: schema.ZodString;
    expiresAt: schema.ZodEffects<schema.ZodDate, Date, Date>;
    scopes: schema.ZodArray<schema.ZodString, "many">;
}, "strip", schema.ZodTypeAny, {
    accessToken: string;
    expiresAt: Date;
    scopes: string[];
}, {
    accessToken: string;
    expiresAt: Date;
    scopes: string[];
}>;
/**
 * This schema represents the format of the session
 * that we cache in the system to avoid unnecessary
 * token exchanges.
 *
 * @example
 * ```
 * {
 *    "accounts.shopify.com": {
 *      "identity": {...} // IdentityTokenSchema
 *      "applications": {
 *        "${domain}-application-id": {  // Admin APIs includes domain in the key
 *          "accessToken": "...",
 *        },
 *        "$application-id": { // ApplicationTokenSchema
 *          "accessToken": "...",
 *        },
 *      }
 *    },
 *    "identity.spin.com": {...}
 * }
 * ```
 */
export declare const SessionSchema: schema.ZodObject<{}, "strip", schema.ZodObject<{
    /**
     * It contains the identity token. Before usint it, we exchange it
     * to get a token that we can use with different applications. The exchanged
     * tokens for the applications are stored under applications.
     */
    identity: schema.ZodObject<{
        accessToken: schema.ZodString;
        refreshToken: schema.ZodString;
        expiresAt: schema.ZodEffects<schema.ZodDate, Date, Date>;
        scopes: schema.ZodArray<schema.ZodString, "many">;
    }, "strip", schema.ZodTypeAny, {
        accessToken: string;
        refreshToken: string;
        expiresAt: Date;
        scopes: string[];
    }, {
        accessToken: string;
        refreshToken: string;
        expiresAt: Date;
        scopes: string[];
    }>;
    /**
     * It contains exchanged tokens for the applications the CLI
     * authenticates with. Tokens are scoped under the fqdn of the applications.
     */
    applications: schema.ZodObject<{}, "strip", schema.ZodObject<{
        accessToken: schema.ZodString;
        expiresAt: schema.ZodEffects<schema.ZodDate, Date, Date>;
        scopes: schema.ZodArray<schema.ZodString, "many">;
    }, "strip", schema.ZodTypeAny, {
        accessToken: string;
        expiresAt: Date;
        scopes: string[];
    }, {
        accessToken: string;
        expiresAt: Date;
        scopes: string[];
    }>, {
        [x: string]: {
            accessToken: string;
            expiresAt: Date;
            scopes: string[];
        };
    }, {
        [x: string]: {
            accessToken: string;
            expiresAt: Date;
            scopes: string[];
        };
    }>;
}, "strip", schema.ZodTypeAny, {
    identity: {
        accessToken: string;
        refreshToken: string;
        expiresAt: Date;
        scopes: string[];
    };
    applications: {
        [x: string]: {
            accessToken: string;
            expiresAt: Date;
            scopes: string[];
        };
    };
}, {
    identity: {
        accessToken: string;
        refreshToken: string;
        expiresAt: Date;
        scopes: string[];
    };
    applications: {
        [x: string]: {
            accessToken: string;
            expiresAt: Date;
            scopes: string[];
        };
    };
}>, {
    [x: string]: {
        identity: {
            accessToken: string;
            refreshToken: string;
            expiresAt: Date;
            scopes: string[];
        };
        applications: {
            [x: string]: {
                accessToken: string;
                expiresAt: Date;
                scopes: string[];
            };
        };
    };
}, {
    [x: string]: {
        identity: {
            accessToken: string;
            refreshToken: string;
            expiresAt: Date;
            scopes: string[];
        };
        applications: {
            [x: string]: {
                accessToken: string;
                expiresAt: Date;
                scopes: string[];
            };
        };
    };
}>;
export type Session = schema.infer<typeof SessionSchema>;
export type IdentityToken = schema.infer<typeof IdentityTokenSchema>;
export type ApplicationToken = schema.infer<typeof ApplicationTokenSchema>;
export {};
