import { buildHeaders, httpsAgent, RequestClientError, sanitizedHeadersOutput } from './headers.js';
import { stringifyMessage, outputContent, outputToken, outputDebug } from '../../../public/node/output.js';
import { AbortError } from '../../../public/node/error.js';
import { debugLogResponseInfo } from '../api.js';
import { ClientError, GraphQLClient } from 'graphql-request';
export function graphqlRequest(query, api, url, token, variables, handleErrors = true) {
    const action = async () => {
        const headers = buildHeaders(token);
        debugLogRequestInfo(api, query, variables, headers);
        const clientOptions = { agent: await httpsAgent(), headers };
        const client = new GraphQLClient(url, clientOptions);
        const response = await debugLogResponseInfo({ request: client.rawRequest(query, variables), url });
        return response.data;
    };
    if (handleErrors) {
        return handlingErrors(api, action);
    }
    else {
        return action();
    }
}
function debugLogRequestInfo(api, query, variables, headers = {}) {
    outputDebug(outputContent `Sending ${outputToken.json(api)} GraphQL request:
  ${outputToken.raw(query.toString().trim())}
${variables ? `\nWith variables:\n${JSON.stringify(variables, null, 2)}\n` : ''}
With request headers:
${sanitizedHeadersOutput(headers)}
`);
}
async function handlingErrors(api, action) {
    try {
        return await action();
    }
    catch (error) {
        if (error instanceof ClientError) {
            const errorMessage = stringifyMessage(outputContent `
  The ${outputToken.raw(api)} GraphQL API responded unsuccessfully with the HTTP status ${`${error.response.status}`} and errors:

  ${outputToken.json(error.response.errors)}
      `);
            let mappedError;
            if (error.response.status < 500) {
                mappedError = new RequestClientError(errorMessage, error.response.status);
            }
            else {
                mappedError = new AbortError(errorMessage);
            }
            mappedError.stack = error.stack;
            throw mappedError;
        }
        else {
            throw error;
        }
    }
}
//# sourceMappingURL=graphql.js.map