import { updateURLsPrompt } from '../../prompts/dev.js';
import { UpdateURLsQuery } from '../../api/graphql/update_urls.js';
import { GetURLsQuery } from '../../api/graphql/get_urls.js';
import { setAppInfo } from '../local-storage.js';
import { AbortError, AbortSilentError, BugError } from '@shopify/cli-kit/node/error';
import { getAvailableTCPPort } from '@shopify/cli-kit/node/tcp';
import { isValidURL } from '@shopify/cli-kit/common/url';
import { partnersRequest } from '@shopify/cli-kit/node/api/partners';
import { appHost, appPort, isSpin, spinFqdn } from '@shopify/cli-kit/node/context/spin';
import { codespaceURL, gitpodURL } from '@shopify/cli-kit/node/context/local';
import { runTunnelPlugin } from '@shopify/cli-kit/node/plugins';
/**
 * The tunnel creation logic depends on 7 variables:
 * - If a Codespaces environment is deteced, then the URL is built using the codespaces hostname. No need for tunnel
 * - If a Gitpod environment is detected, then the URL is built using the gitpod hostname. No need for tunnel
 * - If a Spin environment is detected, then the URL is built using the cli + fqdn hostname as configured in nginx.
 *   No need for tunnel. In case problems with that configuration, the flags Tunnel or Custom Tunnel url could be used
 * - If a tunnelUrl is provided, that takes preference and is returned as the frontendURL
 * - If noTunnel is true, that takes second preference and localhost is used
 * - A Tunnel is created then if any of these conditions are met:
 *   - Tunnel flag is true
 *   - The app has UI extensions
 *   - In a previous run, the user selected to always use a tunnel (cachedTunnelPlugin)
 * - Otherwise, localhost is used
 *
 * If there is no cached tunnel plugin and a tunnel is necessary, we'll ask the user to confirm.
 */
export async function generateFrontendURL(options) {
    let frontendPort = 4040;
    let frontendUrl;
    let usingLocalhost = false;
    const hasExtensions = options.app.hasUIExtensions();
    const needsTunnel = (hasExtensions || options.tunnel || options.cachedTunnelPlugin) && !options.noTunnel;
    if (codespaceURL()) {
        frontendUrl = `https://${codespaceURL()}-${frontendPort}.githubpreview.dev`;
        return { frontendUrl, frontendPort, usingLocalhost };
    }
    if (gitpodURL()) {
        const defaultUrl = gitpodURL()?.replace('https://', '');
        frontendUrl = `https://${frontendPort}-${defaultUrl}`;
        return { frontendUrl, frontendPort, usingLocalhost };
    }
    if (isSpin() && !options.tunnelUrl) {
        frontendUrl = `https://cli.${await spinFqdn()}`;
        if (appPort() !== undefined) {
            frontendPort = appPort() ?? frontendPort;
            frontendUrl = `https://${appHost()}`;
        }
        return { frontendUrl, frontendPort, usingLocalhost };
    }
    if (options.tunnelUrl) {
        const matches = options.tunnelUrl.match(/(https:\/\/[^:]+):([0-9]+)/);
        if (!matches) {
            throw new AbortError(`Invalid tunnel URL: ${options.tunnelUrl}`, 'Valid format: "https://my-tunnel-url:port"');
        }
        frontendPort = Number(matches[2]);
        frontendUrl = matches[1];
        return { frontendUrl, frontendPort, usingLocalhost };
    }
    if (needsTunnel) {
        frontendPort = await getAvailableTCPPort();
        frontendUrl = await generateURL(options.commandConfig, frontendPort);
    }
    else {
        frontendPort = await getAvailableTCPPort();
        frontendUrl = 'http://localhost';
        usingLocalhost = true;
    }
    return { frontendUrl, frontendPort, usingLocalhost };
}
export async function generateURL(config, frontendPort) {
    // For the moment we assume to always have ngrok, this will change in a future PR
    // and will need to use "getListOfTunnelPlugins" to find the available tunnel plugins
    const provider = 'ngrok';
    return (await runTunnelPlugin(config, frontendPort, provider)).mapError(mapRunTunnelPluginError).valueOrAbort();
}
export function generatePartnersURLs(baseURL, authCallbackPath) {
    let redirectUrlWhitelist;
    if (authCallbackPath && authCallbackPath.length > 0) {
        const authCallbackPaths = Array.isArray(authCallbackPath) ? authCallbackPath : [authCallbackPath];
        redirectUrlWhitelist = authCallbackPaths.reduce((acc, path) => {
            if (path && path.length > 0) {
                acc.push(`${baseURL}${path}`);
            }
            return acc;
        }, []);
    }
    else {
        redirectUrlWhitelist = [
            `${baseURL}/auth/callback`,
            `${baseURL}/auth/shopify/callback`,
            `${baseURL}/api/auth/callback`,
        ];
    }
    return {
        applicationUrl: baseURL,
        redirectUrlWhitelist,
    };
}
export async function updateURLs(urls, apiKey, token) {
    const variables = { apiKey, ...urls };
    const query = UpdateURLsQuery;
    const result = await partnersRequest(query, token, variables);
    if (result.appUpdate.userErrors.length > 0) {
        const errors = result.appUpdate.userErrors.map((error) => error.message).join(', ');
        throw new AbortError(errors);
    }
}
export async function getURLs(apiKey, token) {
    const variables = { apiKey };
    const query = GetURLsQuery;
    const result = await partnersRequest(query, token, variables);
    return { applicationUrl: result.app.applicationUrl, redirectUrlWhitelist: result.app.redirectUrlWhitelist };
}
export async function shouldOrPromptUpdateURLs(options) {
    if (options.newApp)
        return true;
    let shouldUpdate = options.cachedUpdateURLs === true;
    if (options.cachedUpdateURLs === undefined) {
        const response = await updateURLsPrompt(options.currentURLs.applicationUrl, options.currentURLs.redirectUrlWhitelist);
        let newUpdateURLs;
        /* eslint-disable no-fallthrough */
        switch (response) {
            case 'always':
                newUpdateURLs = true;
            case 'yes':
                shouldUpdate = true;
                break;
            case 'never':
                newUpdateURLs = false;
            case 'no':
                shouldUpdate = false;
        }
        /* eslint-enable no-fallthrough */
        setAppInfo({ directory: options.appDirectory, updateURLs: newUpdateURLs });
    }
    return shouldUpdate;
}
export function validatePartnersURLs(urls) {
    if (!isValidURL(urls.applicationUrl))
        throw new AbortError(`Invalid application URL: ${urls.applicationUrl}`, 'Valid format: "https://example.com"');
    urls.redirectUrlWhitelist.forEach((url) => {
        if (!isValidURL(url))
            throw new AbortError(`Invalid redirection URLs: ${urls.redirectUrlWhitelist}`, 'Valid format: "https://example.com/callback1,https://example.com/callback2"');
    });
}
function mapRunTunnelPluginError(tunnelPluginError) {
    switch (tunnelPluginError.type) {
        case 'no-provider':
            return new BugError(`We couldn't find the ${tunnelPluginError.provider} tunnel plugin`);
        case 'multiple-urls':
            return new BugError('Multiple tunnel plugins for ngrok found');
        case 'unknown':
            return new BugError(`${tunnelPluginError.provider} failed to start the tunnel.\n${tunnelPluginError.message}`);
        default:
            return new AbortSilentError();
    }
}
//# sourceMappingURL=urls.js.map