import { addressPrompt, apiVersionPrompt, deliveryMethodInstructions, deliveryMethodPrompt, clientSecretPrompt, topicPrompt, } from './trigger.js';
import { DELIVERY_METHOD, deliveryMethodForAddress, isAddressAllowedForDeliveryMethod, } from '../../services/webhook/trigger-options.js';
import { AbortError } from '@shopify/cli-kit/node/error';
export async function collectApiVersion(apiVersion, availableVersions) {
    const apiVersionPassed = flagPassed(apiVersion);
    if (apiVersionPassed) {
        const passedApiVersion = apiVersion.trim();
        if (availableVersions.includes(passedApiVersion)) {
            return passedApiVersion;
        }
        throw new AbortError(`Api Version '${passedApiVersion}' does not exist`, `Allowed values: ${availableVersions.join(', ')}`, ['Try again with a valid api-version value']);
    }
    const promptedApiVersion = await apiVersionPrompt(availableVersions);
    return promptedApiVersion;
}
export async function collectTopic(topic, apiVersion, availableTopics) {
    const topicPassed = flagPassed(topic);
    if (topicPassed) {
        const passedTopic = equivalentTopic(topic.trim(), availableTopics);
        if (passedTopic === undefined) {
            throw new AbortError(`Topic '${passedTopic}' does not exist for ApiVersion '${apiVersion}'`, `Allowed values: ${availableTopics.join(', ')}`, ['Try again with a valid api-version - topic pair']);
        }
        return passedTopic;
    }
    if (availableTopics.length === 0) {
        throw new AbortError(`No topics found for '${apiVersion}'`);
    }
    const promptedTopic = await topicPrompt(availableTopics);
    return promptedTopic;
}
export async function collectAddressAndMethod(deliveryMethod, address) {
    const methodWasPassed = flagPassed(deliveryMethod);
    const addressWasPassed = flagPassed(address);
    if (methodWasPassed && !validDeliveryMethodFlag(deliveryMethod)) {
        throw new AbortError('Invalid Delivery Method passed', `${DELIVERY_METHOD.HTTP}, ${DELIVERY_METHOD.PUBSUB}, and ${DELIVERY_METHOD.EVENTBRIDGE} are allowed`, ['Try again with a valid delivery method']);
    }
    // Method is valid
    let actualAddress = '';
    let actualMethod = '';
    if (methodWasPassed && addressWasPassed) {
        if (isAddressAllowedForDeliveryMethod(address, deliveryMethod)) {
            actualAddress = address.trim();
            actualMethod = inferMethodFromAddress(actualAddress);
        }
        else {
            throw new AbortError(`Can't deliver your webhook payload to this address using '${deliveryMethod}'`, "Run 'shopify webhook trigger --address=<VALUE>' with a valid URL", deliveryMethodInstructions(deliveryMethod));
        }
    }
    if (!methodWasPassed && addressWasPassed) {
        actualAddress = address.trim();
        actualMethod = inferMethodFromAddress(actualAddress);
    }
    if (methodWasPassed && !addressWasPassed) {
        actualAddress = await addressPrompt(deliveryMethod);
        actualMethod = inferMethodFromAddress(actualAddress);
    }
    if (!methodWasPassed && !addressWasPassed) {
        const method = await deliveryMethodPrompt();
        actualAddress = await addressPrompt(method);
        actualMethod = inferMethodFromAddress(actualAddress);
    }
    return [actualMethod, actualAddress];
}
export async function collectSecret(clientSecret) {
    const secret = await useFlagOrPrompt(clientSecret, clientSecretPrompt);
    return secret;
}
async function useFlagOrPrompt(value, prompt) {
    return flagPassed(value) ? value : prompt();
}
function flagPassed(flag) {
    if (flag === undefined) {
        return false;
    }
    return flag.length > 0;
}
function validDeliveryMethodFlag(value) {
    return value === DELIVERY_METHOD.HTTP || value === DELIVERY_METHOD.PUBSUB || value === DELIVERY_METHOD.EVENTBRIDGE;
}
function inferMethodFromAddress(address) {
    const method = deliveryMethodForAddress(address);
    if (method === undefined) {
        throw new AbortError('No delivery method available for the address', `Use a valid address for ${DELIVERY_METHOD.HTTP}, ${DELIVERY_METHOD.PUBSUB} or ${DELIVERY_METHOD.EVENTBRIDGE}`);
    }
    return method;
}
function equivalentTopic(passedTopic, availableTopics) {
    if (availableTopics.includes(passedTopic)) {
        return passedTopic;
    }
    return availableTopics.find((elm) => elm.toUpperCase().replace('/', '_') === passedTopic);
}
//# sourceMappingURL=options-prompt.js.map